package com.demoscene.tv

import android.os.Bundle
import android.view.KeyEvent
import android.webkit.WebChromeClient
import android.webkit.WebSettings
import android.webkit.WebView
import android.webkit.WebViewClient
import androidx.fragment.app.FragmentActivity
import com.demoscene.tv.data.DemoDataProvider

class PlaybackActivity : FragmentActivity() {
    
    companion object {
        const val EXTRA_DEMO_ID = "demo_id"
    }
    
    private lateinit var webView: WebView
    
    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)
        
        val demoId = intent.getStringExtra(EXTRA_DEMO_ID) ?: run {
            finish()
            return
        }
        
        val demo = DemoDataProvider.getDemoById(demoId) ?: run {
            finish()
            return
        }
        
        val videoId = demo.getYoutubeVideoId() ?: run {
            android.widget.Toast.makeText(
                this,
                "No YouTube link available",
                android.widget.Toast.LENGTH_SHORT
            ).show()
            finish()
            return
        }
        
        setupWebView(videoId)
    }
    
    private fun setupWebView(videoId: String) {
        webView = WebView(this)
        setContentView(webView)
        
        // Configure WebView for YouTube
        webView.settings.apply {
            javaScriptEnabled = true
            domStorageEnabled = true
            mediaPlaybackRequiresUserGesture = false
            cacheMode = WebSettings.LOAD_NO_CACHE
        }
        
        webView.webChromeClient = WebChromeClient()
        webView.webViewClient = WebViewClient()
        
        // Load YouTube embedded player
        val htmlContent = """
            <!DOCTYPE html>
            <html>
            <head>
                <meta name="viewport" content="width=device-width, initial-scale=1.0">
                <style>
                    * {
                        margin: 0;
                        padding: 0;
                        overflow: hidden;
                    }
                    body, html {
                        height: 100%;
                        background-color: #000;
                    }
                    iframe {
                        position: absolute;
                        top: 0;
                        left: 0;
                        width: 100%;
                        height: 100%;
                        border: 0;
                    }
                </style>
            </head>
            <body>
                <iframe 
                    src="https://www.youtube.com/embed/$videoId?autoplay=1&controls=1&modestbranding=1&rel=0&showinfo=0&fs=1&iv_load_policy=3"
                    frameborder="0" 
                    allow="accelerometer; autoplay; clipboard-write; encrypted-media; gyroscope; picture-in-picture; web-share" 
                    allowfullscreen>
                </iframe>
            </body>
            </html>
        """.trimIndent()
        
        webView.loadDataWithBaseURL(
            "https://www.youtube.com",
            htmlContent,
            "text/html",
            "UTF-8",
            null
        )
    }
    
    override fun onKeyDown(keyCode: Int, event: KeyEvent?): Boolean {
        // Handle back button to exit playback
        if (keyCode == KeyEvent.KEYCODE_BACK) {
            finish()
            return true
        }
        return super.onKeyDown(keyCode, event)
    }
    
    override fun onDestroy() {
        webView.destroy()
        super.onDestroy()
    }
}
