
// ------------------------------------------------------------------------------------------------
//  MetaBalls.cpp - MetaBalls with Marching Cubes.
//  Copyright (C) 2001. Remage / Fresh!mindworkz.
// ------------------------------------------------------------------------------------------------

#include <StdIO.h>
#include <Windows.h>
#include <D3D8.h>
#include "D3DU8.h"

#include "Main.h"
#include "Fmath.h"
#include "TxGen.h"
#include "Rand.h"

#include "PolyTab.h"

#define GRID (32)
#define GRIDS (1024)
#define GRID2 (16.0f)

#define METABALLS_NEGATIVE  0x01

// ------------------------------------------------------------------------------------------------

extern int ExportTGA;

static int METABALLS = 5;

static float *MetaBalls_Field = NULL;
static int MetaBalls_ActFrame = 0, *MetaBalls_Frame = NULL;
static float MetaBalls_Max, MetaBalls_Max1, MetaBalls_Max2;

D3DVECTOR MetaBalls_Position[ 8 ];

#define D3DFVF_METABALLS_VERTEX1 (D3DFVF_XYZ|D3DFVF_NORMAL)
typedef struct _METABALLS_VERTEX1
  {
    D3DVECTOR Vertex, Normal;
   } METABALLS_VERTEX1;

LPDIRECT3DVERTEXBUFFER8 MetaBalls_VertexBuffer1 = NULL;
METABALLS_VERTEX1 *MetaBalls_LockedVertexBuffer1;
DWORD MetaBalls_VertexIdx1 = 0;

LPDIRECT3DTEXTURE8 MetaBalls_Texture1 = NULL, MetaBalls_Texture2 = NULL, MetaBalls_Texture3 = NULL, MetaBalls_Texture4 = NULL;
D3DMATRIX MetaBalls_TextureMatrix0 = 
  { 0.5f, 0.0f, 0.0f, 0.0f, 
    0.0f, 0.5f, 0.0f, 0.0f,
    0.0f, 0.0f, 1.0f, 0.0f,
    0.5f, 0.5f, 0.0f, 1.0f };

DWORD MetaBalls_MaxPolygon = 0;

D3DMATERIAL8 MetaBalls_Material1;
D3DLIGHT8 MetaBalls_Light1;

static TXGEN_LAYER *TxLayer1, *TxLayer2, *TxLayer3;

// ------------------------------------------------------------------------------------------------

int MetaBalls_Initialize( void )
  {
    int I, J, K;

    MetaBalls_Field = (float*) HeapAlloc( GetProcessHeap(), HEAP_ZERO_MEMORY, GRID * GRID * GRID * sizeof( float ));
    MetaBalls_Frame = (int*) HeapAlloc( GetProcessHeap(), HEAP_ZERO_MEMORY, GRID * GRID * GRID * sizeof( int ));

    // --- Create VertexBuffer.
    if ( Direct3DDevice->CreateVertexBuffer( 3 * 12288 * sizeof( METABALLS_VERTEX1 ), D3DUSAGE_DYNAMIC, 
      D3DFVF_METABALLS_VERTEX1, D3DPOOL_DEFAULT, &MetaBalls_VertexBuffer1 ) != D3D_OK ) return 0;

    // --- Generate Textures.
    TxLayer1 = TxGen_LayerInitialize( 512, 512 );
    TxLayer2 = TxGen_LayerInitialize( 512, 512 );
    TxLayer3 = TxGen_LayerInitialize( 512, 512 );

    // --- TextueMap1.
    Direct3DDevice->CreateTexture( 512, 512, 1, 0, D3DFMT_X8R8G8B8, D3DPOOL_DEFAULT, &MetaBalls_Texture1 );
    if ( !TxGen_LoadTGA( MetaBalls_Texture1, "Dreams02.tga" ))
      {
        TxGen_SubPlasma( TxLayer1, 0x463F7B, 0xD6C1FF, 7, 3 );
        TxGen_SubPlasma( TxLayer2, 0x463F7B, 0xD6C1FF, 8, 6 );
        TxGen_LayerBlend( TxLayer1, TxLayer1, TxLayer2, 50, TXGEN_BLEND_NORMAL );
        TxGen_SubPlasma( TxLayer2, 0x463F7B, 0xD6C1FF, 9, 12 );
        TxGen_LayerBlend( TxLayer1, TxLayer1, TxLayer2, 25, TXGEN_BLEND_NORMAL );
        TxGen_SubPlasma( TxLayer2, 0x282F3D, 0x6289B4, 12, 4 );
        TxGen_SubPlasma( TxLayer3, 0x282F3D, 0x6289B4, 13, 16 );
        TxGen_LayerBlend( TxLayer2, TxLayer2, TxLayer3, 40, TXGEN_BLEND_NORMAL );
        TxGen_SubPlasma( TxLayer3, 0x282F3D, 0x6289B4, 14, 32 );
        TxGen_LayerBlend( TxLayer2, TxLayer2, TxLayer3, 20, TXGEN_BLEND_NORMAL );
        TxGen_LayerBlend( TxLayer1, TxLayer1, TxLayer2, 100, TXGEN_BLEND_CUSTOM );
        TxGen_BrightnessContrast( TxLayer1, -40, 0 );
        TxGen_SetTexture( MetaBalls_Texture1, TxLayer1 );
        if ( ExportTGA ) TxGen_SaveLayerTGA( TxLayer1, "Dreams02.tga" );
       }

    // --- TextueMap2.
    Direct3DDevice->CreateTexture( 512, 512, 1, 0, D3DFMT_X8R8G8B8, D3DPOOL_DEFAULT, &MetaBalls_Texture4 );
    if ( !TxGen_LoadTGA( MetaBalls_Texture4, "Dreams03.tga" ))
      {
        TxGen_SubPlasma( TxLayer1, 0x283D39, 0x76B9AD, 15, 3 );
        TxGen_SubPlasma( TxLayer2, 0x283D39, 0x76B9AD, 16, 6 );
        TxGen_LayerBlend( TxLayer1, TxLayer1, TxLayer2, 50, TXGEN_BLEND_NORMAL );
        TxGen_SubPlasma( TxLayer2, 0x283D39, 0x76B9AD, 17, 12 );
        TxGen_LayerBlend( TxLayer1, TxLayer1, TxLayer2, 25, TXGEN_BLEND_NORMAL );
        TxGen_SubPlasma( TxLayer2, 0x282F3D, 0x6289B4, 18, 4 );
        TxGen_SubPlasma( TxLayer3, 0x282F3D, 0x6289B4, 19, 16 );
        TxGen_LayerBlend( TxLayer2, TxLayer2, TxLayer3, 40, TXGEN_BLEND_NORMAL );
        TxGen_SubPlasma( TxLayer3, 0x282F3D, 0x6289B4, 20, 32 );
        TxGen_LayerBlend( TxLayer2, TxLayer2, TxLayer3, 20, TXGEN_BLEND_NORMAL );
        TxGen_LayerBlend( TxLayer1, TxLayer1, TxLayer2, 100, TXGEN_BLEND_CUSTOM );
        TxGen_BrightnessContrast( TxLayer1, -20, 0 );
        TxGen_SetTexture( MetaBalls_Texture4, TxLayer1 );
        if ( ExportTGA ) TxGen_SaveLayerTGA( TxLayer1, "Dreams03.tga" );
       }

    // --- EnvMap1.
    Direct3DDevice->CreateTexture( 512, 512, 1, 0, D3DFMT_X8R8G8B8, D3DPOOL_DEFAULT, &MetaBalls_Texture2 );
    if ( !TxGen_LoadTGA( MetaBalls_Texture2, "Dreams04.tga" ))
      {
        TxGen_SubPlasma( TxLayer1, 0x231F26, 0xFFFFFF, 0, 5 );
        TxGen_EnvMap( TxLayer2, 0x1F1F1F, 0xFFFFFF, 255 );
        TxGen_LayerBlend( TxLayer1, TxLayer1, TxLayer2, 100, TXGEN_BLEND_MULTIPLY );
        TxGen_SubPlasma( TxLayer2, 0x000000, 0x463F4B, 54, 11 );
        TxGen_LayerBlend( TxLayer1, TxLayer1, TxLayer2, 50, TXGEN_BLEND_ADD );
        TxGen_SubPlasma( TxLayer2, 0x000000, 0x463F4B, 53, 31 );
        TxGen_LayerBlend( TxLayer1, TxLayer1, TxLayer2, 10, TXGEN_BLEND_ADD );
        TxGen_SetTexture( MetaBalls_Texture2, TxLayer1 );
        if ( ExportTGA ) TxGen_SaveLayerTGA( TxLayer1, "Dreams04.tga" );
       }

    // --- EnvMap2.
    Direct3DDevice->CreateTexture( 512, 512, 1, 0, D3DFMT_X8R8G8B8, D3DPOOL_DEFAULT, &MetaBalls_Texture3 );
    if ( !TxGen_LoadTGA( MetaBalls_Texture3, "Dreams05.tga" ))
      {
        TxGen_SubPlasma( TxLayer1, 0x463F4B, 0x8F8F8F, 1, 4 );
        TxGen_SubPlasma( TxLayer2, 0x000000, 0x8F8F8F, 2, 8 );
        TxGen_LayerBlend( TxLayer1, TxLayer1, TxLayer2, 50, TXGEN_BLEND_ADD );
        TxGen_SubPlasma( TxLayer2, 0x000000, 0x6F6F6F, 3, 16 );
        TxGen_LayerBlend( TxLayer1, TxLayer1, TxLayer2, 20, TXGEN_BLEND_ADD );
        TxGen_SubPlasma( TxLayer2, 0x000000, 0x5F5F6F, 4, 32 );
        TxGen_LayerBlend( TxLayer1, TxLayer1, TxLayer2, 10, TXGEN_BLEND_ADD );
        TxGen_EnvMap( TxLayer2, 0x1F1F1F, 0xCFCFCF, 255 );
        TxGen_LayerBlend( TxLayer1, TxLayer1, TxLayer2, 100, TXGEN_BLEND_MULTIPLY );
        TxGen_SetTexture( MetaBalls_Texture3, TxLayer1 );
        if ( ExportTGA ) TxGen_SaveLayerTGA( TxLayer1, "Dreams05.tga" );
       }

    TxGen_LayerCleanup( TxLayer1 );
    TxGen_LayerCleanup( TxLayer2 );
    TxGen_LayerCleanup( TxLayer3 );

    for ( I = 0; I < GRID; I++ )
      for ( J = 0; J < GRID; J++ )
        for ( K = 0; K < GRID; K++ )
          MetaBalls_Field[ I*GRIDS + J*GRID + K ] = 0.0f;

    return 1;
   }

// ------------------------------------------------------------------------------------------------

void MetaBalls_Cleanup( void )
  {
    if ( MetaBalls_Field ) HeapFree( GetProcessHeap(), 0, (void*) MetaBalls_Field );
    if ( MetaBalls_Frame ) HeapFree( GetProcessHeap(), 0, (void*) MetaBalls_Frame );

    if ( MetaBalls_VertexBuffer1 ) MetaBalls_VertexBuffer1->Release();
    if ( MetaBalls_Texture1 ) MetaBalls_Texture1->Release();
    if ( MetaBalls_Texture2 ) MetaBalls_Texture2->Release();
    if ( MetaBalls_Texture3 ) MetaBalls_Texture3->Release();
    if ( MetaBalls_Texture4 ) MetaBalls_Texture4->Release();
/*
    char Tmp[64];
    sprintf( Tmp, "MetaBalls_MaxPolygon == %d", MetaBalls_MaxPolygon );
    MessageBox( 0, Tmp, "Debug", MB_OK );
//*/
   }

// ------------------------------------------------------------------------------------------------

void MetaBalls_MarchingCubeInterpolate( int X, int Y, int Z, int VertexIdx, float IsoValue, unsigned char PolyFlags )
  {
    int DX1, DY1, DZ1, DX2, DY2, DZ2;
    float X0, Y0, Z0, NX0, NY0, NZ0, NX1, NY1, NZ1, NX2, NY2, NZ2, DB;

    DX2 = X + (( PolyFlags >> 6 ) & 1 );
    DY2 = Y + (( PolyFlags >> 5 ) & 1 );
    DZ2 = Z + (( PolyFlags >> 4 ) & 1 );
    DX1 = X + (( PolyFlags >> 2 ) & 1 );
    DY1 = Y + (( PolyFlags >> 1 ) & 1 );
    DZ1 = Z + (( PolyFlags >> 0 ) & 1 );

    DB = 1.0f / ( MetaBalls_Field[ DX2*GRIDS + DY2*GRID + DZ2 ] - MetaBalls_Field[ DX1*GRIDS + DY1*GRID + DZ1 ] );
    DB *= ( IsoValue - MetaBalls_Field[ DX1*GRIDS + DY1*GRID + DZ1 ] );

    if ( DX1 == DX2 )
      X0 = (float) DX1; else
      X0 = (float) DX1 + (float) ( DX2 - DX1 ) * DB;
    if ( DY1 == DY2 )
      Y0 = (float) DY1; else
      Y0 = (float) DY1 + (float) ( DY2 - DY1 ) * DB;
    if ( DZ1 == DZ2 )
      Z0 = (float) DZ1; else
      Z0 = (float) DZ1 + (float) ( DZ2 - DZ1 ) * DB;

    NX1 = MetaBalls_Field[ (DX1+1)*GRIDS + DY1*GRID + DZ1 ] - MetaBalls_Field[ (DX1-1)*GRIDS + DY1*GRID + DZ1 ];
    NY1 = MetaBalls_Field[ DX1*GRIDS + (DY1+1)*GRID + DZ1 ] - MetaBalls_Field[ DX1*GRIDS + (DY1-1)*GRID + DZ1 ];
    NZ1 = MetaBalls_Field[ DX1*GRIDS + DY1*GRID + (DZ1+1) ] - MetaBalls_Field[ DX1*GRIDS + DY1*GRID + (DZ1-1) ];

    NX2 = MetaBalls_Field[ (DX2+1)*GRIDS + DY2*GRID + DZ2 ] - MetaBalls_Field[ (DX2-1)*GRIDS + DY2*GRID + DZ2 ];
    NY2 = MetaBalls_Field[ DX2*GRIDS + (DY2+1)*GRID + DZ2 ] - MetaBalls_Field[ DX2*GRIDS + (DY2-1)*GRID + DZ2 ];
    NZ2 = MetaBalls_Field[ DX2*GRIDS + DY2*GRID + (DZ2+1) ] - MetaBalls_Field[ DX2*GRIDS + DY2*GRID + (DZ2-1) ];

    NX0 = (float) NX1 + (float) ( NX2 - NX1 ) * DB;
    NY0 = (float) NY1 + (float) ( NY2 - NY1 ) * DB;
    NZ0 = (float) NZ1 + (float) ( NZ2 - NZ1 ) * DB;

    D3DUVec3Create( &MetaBalls_LockedVertexBuffer1[ VertexIdx ].Vertex, X0 - GRID2, Y0 - GRID2, Z0 - GRID2 );
    D3DUVec3Create( &MetaBalls_LockedVertexBuffer1[ VertexIdx ].Normal, NX0, NY0, NZ0 );
    D3DUVec3Normalize( &MetaBalls_LockedVertexBuffer1[ VertexIdx ].Normal, &MetaBalls_LockedVertexBuffer1[ VertexIdx ].Normal );
   }

// ------------------------------------------------------------------------------------------------

void MetaBalls_MarchingCubes( float IsoValue, DWORD Flags )
  {
    int X, Y, Z, Ofs, Cube, Pt;

    // --- Update VertexBuffer.
    if ( MetaBalls_VertexBuffer1->Lock( 0, 3*12288*sizeof( METABALLS_VERTEX1 ), (BYTE**) &MetaBalls_LockedVertexBuffer1, D3DLOCK_DISCARD ) != D3D_OK ) return;
    MetaBalls_VertexIdx1 = 0;

    for ( X = 1; X < GRID-2; X++ )
      for ( Y = 1; Y < GRID-2; Y++ )
        for ( Z = 1; Z < GRID-2; Z++ )
          {
            Ofs = X*GRIDS + Y*GRID + Z;
            if ( MetaBalls_Frame[ Ofs ] == MetaBalls_ActFrame )
              {
                if ( Flags & METABALLS_NEGATIVE )
                  {
                    Cube = 
                      ((( MetaBalls_Field[ Ofs ] < -IsoValue ) ? 1:0 ) +
                       (( MetaBalls_Field[ Ofs+1 ] < -IsoValue ) ? 2:0 ) +
                       (( MetaBalls_Field[ Ofs+GRID ] < -IsoValue ) ? 4:0 ) +
                       (( MetaBalls_Field[ Ofs+GRID+1 ] < -IsoValue ) ? 8:0 ) +
                       (( MetaBalls_Field[ Ofs+GRIDS ] < -IsoValue ) ? 16:0 ) +
                       (( MetaBalls_Field[ Ofs+GRIDS+1 ] < -IsoValue ) ? 32:0 ) +
                       (( MetaBalls_Field[ Ofs+GRIDS+GRID ] < -IsoValue ) ? 64:0 )) +
                       (( MetaBalls_Field[ Ofs+GRIDS+GRID+1 ] < -IsoValue ) ? 128:0 );
                    if ( Cube-- )
                      {
                        Pt = 0;
                        Cube <<= 4;
                        while ( PolyTab[ Cube+Pt ] > 0 )
                          {
                            MetaBalls_MarchingCubeInterpolate( X, Y, Z, MetaBalls_VertexIdx1+0, -IsoValue, PolyTab[ Cube+Pt+0 ] );
                            MetaBalls_MarchingCubeInterpolate( X, Y, Z, MetaBalls_VertexIdx1+1, -IsoValue, PolyTab[ Cube+Pt+1 ] );
                            MetaBalls_MarchingCubeInterpolate( X, Y, Z, MetaBalls_VertexIdx1+2, -IsoValue, PolyTab[ Cube+Pt+2 ] );
                            MetaBalls_VertexIdx1 += 3;
                            Pt += 3;
                           }
                       }
                   } else
                  {
                    Cube = 
                      ((( MetaBalls_Field[ Ofs ] > IsoValue ) ? 1:0 ) +
                       (( MetaBalls_Field[ Ofs+1 ] > IsoValue ) ? 2:0 ) +
                       (( MetaBalls_Field[ Ofs+GRID ] > IsoValue ) ? 4:0 ) +
                       (( MetaBalls_Field[ Ofs+GRID+1 ] > IsoValue ) ? 8:0 ) +
                       (( MetaBalls_Field[ Ofs+GRIDS ] > IsoValue ) ? 16:0 ) +
                       (( MetaBalls_Field[ Ofs+GRIDS+1 ] > IsoValue ) ? 32:0 ) +
                       (( MetaBalls_Field[ Ofs+GRIDS+GRID ] > IsoValue ) ? 64:0 )) +
                       (( MetaBalls_Field[ Ofs+GRIDS+GRID+1 ] > IsoValue ) ? 128:0 );
                    if ( Cube-- )
                      {
                        Pt = 0;
                        Cube <<= 4;
                        while ( PolyTab[ Cube+Pt ] > 0 )
                          {
                            MetaBalls_MarchingCubeInterpolate( X, Y, Z, MetaBalls_VertexIdx1+0, IsoValue, PolyTab[ Cube+Pt+0 ] );
                            MetaBalls_MarchingCubeInterpolate( X, Y, Z, MetaBalls_VertexIdx1+1, IsoValue, PolyTab[ Cube+Pt+1 ] );
                            MetaBalls_MarchingCubeInterpolate( X, Y, Z, MetaBalls_VertexIdx1+2, IsoValue, PolyTab[ Cube+Pt+2 ] );
                            MetaBalls_VertexIdx1 += 3;
                            Pt += 3;
                           }
                       }
                   }
               }
           }
    MetaBalls_VertexBuffer1->Unlock();
   }

// ------------------------------------------------------------------------------------------------

void MetaBalls_Render1( float Time )
  {
    int I, J, K, MB;
    float Dist, TimeDist;

    METABALLS = 4;

    MetaBalls_ActFrame++;

    MetaBalls_Max = 6.0f;
    MetaBalls_Max1 = Fsqr( MetaBalls_Max );
    MetaBalls_Max2 = 1.0f / MetaBalls_Max1;

    for ( I = 0; I < GRID; I++ )
      for ( J = 0; J < GRID; J++ )
        for ( K = 0; K < GRID; K++ )
          {
            MetaBalls_Field[ I*GRIDS + J*GRID + K ] *= 0.9f;
            MetaBalls_Frame[ I*GRIDS + J*GRID + K ] = MetaBalls_ActFrame;
           }
//        MetaBalls_Field[ I*GRIDS + J*GRID + K ] = 0.0f;

    TimeDist = 0.0f;
    if ( Time > 560.0f ) TimeDist = ( Time - 560.0f ) / 32.0f;
    if ( Time > 592.0f ) TimeDist = 1.0f;
    if ( Time > 672 ) TimeDist = ( 704.0f - Time ) / 32.0f;
    if ( Time > 704 ) TimeDist = 0.0f;

    for ( MB = 0; MB < METABALLS; MB++ )
      {
        D3DUVec3Create( &MetaBalls_Position[ MB ],
          GRID2 + TimeDist * ( 5.0f + 2.0f * Fsin( Time * (float) ( MB+5 ) * 0.019f )) * Fcos(( (float) MB / 8.0f ) + Time * (float) ( MB+2 ) * 0.015f ),
          GRID2 + TimeDist * ( 5.0f + 2.0f * Fsin( Time * (float) ( MB+3 ) * 0.023f )) * Fsin(( (float) MB / 8.0f ) + Time * (float) ( MB+3 ) * 0.016f ),
          GRID2 + TimeDist * 7.0f * Fcos(( (float) MB / 8.0f ) + Time * (float) ( MB+5 ) * 0.013f ));
          
          for ( I = 0; I < GRID; I++ )
            for ( J = 0; J < GRID; J++ )
              for ( K = 0; K < GRID; K++ )
                {
                  Dist = Fsqr( I - MetaBalls_Position[ MB ].x ) + Fsqr( J - MetaBalls_Position[ MB ].y ) + Fsqr( K - MetaBalls_Position[ MB ].z );
                  if ( Dist < MetaBalls_Max1 )
                    {
/*
                      if ( !MB )
                        MetaBalls_Field[ I*GRIDS + J*GRID + K ] -= 0.4f * Fsqr( 1.0f - Dist * MetaBalls_Max2 ); else
//*/
                        MetaBalls_Field[ I*GRIDS + J*GRID + K ] += 0.6f * Fsqr( 1.0f - Dist * MetaBalls_Max2 );
                      MetaBalls_Frame[ I*GRIDS + J*GRID + K ] = MetaBalls_ActFrame;
                     }
                 }
       }

    MetaBalls_MarchingCubes( 47.0f * Fsqr( 1.0f - TimeDist ) + 0.7f + 0.2f * Fsin( Time * 0.1f ), 0 );

    // --- Setup World Matrix.
    Direct3DDevice->SetTransform( D3DTS_WORLD, &D3DUMatrixIdentity );

    // --- Setup View Matrix.
    D3DUVec3Create( &D3DVector_CameraPosition, 26.0f, 26.0f, -10.0f );
    D3DUVec3Create( &D3DVector_CameraTarget, 10.0f, 8.0f, 0.0f );
    D3DUVec3Create( &D3DVector_Up, 0.0f, -1.0f, 0.0f );
    D3DUMatrixLookAtLH( &D3DMatrix_View, &D3DVector_CameraPosition, &D3DVector_CameraTarget, &D3DVector_Up );
    Direct3DDevice->SetTransform( D3DTS_VIEW, &D3DMatrix_View );

    // --- Setup Projection Matrix.
    D3DUMatrixPerspectiveFovLH( &D3DMatrix_Projection, D3DU_PI * 0.25f, 1.778f, 0.1f, 100.0f );
    Direct3DDevice->SetTransform( D3DTS_PROJECTION, &D3DMatrix_Projection );

    // --- Setup Texture Matrix.
    D3DUMatrixTranslation( &D3DMatrix_Texture0, 0.3f * Fsin( Time * 0.06f ), 0.3f * Fcos( Time * 0.034f ), 0.3f * Fsin( Time * 0.047f ));
    D3DUMatrixRotationZ( &D3DMatrix_Temp, Time * 0.1f );
    D3DUMatrixMultiply( &D3DMatrix_Texture0, &D3DMatrix_Texture0, &D3DMatrix_Temp );
    D3DUMatrixMultiply( &D3DMatrix_Texture0, &D3DMatrix_Texture0, &MetaBalls_TextureMatrix0 );
    Direct3DDevice->SetTransform( D3DTS_TEXTURE0, &D3DMatrix_Texture0 );
    Direct3DDevice->SetTransform( D3DTS_TEXTURE1, &MetaBalls_TextureMatrix0 );

    // --- Setup Texture.
    Direct3DDevice->SetTexture( 0, MetaBalls_Texture1 );
    Direct3DDevice->SetTextureStageState( 0, D3DTSS_TEXTURETRANSFORMFLAGS, D3DTTFF_COUNT2 );
    Direct3DDevice->SetTextureStageState( 0, D3DTSS_TEXCOORDINDEX, D3DTSS_TCI_CAMERASPACENORMAL );

    Direct3DDevice->SetTextureStageState( 0, D3DTSS_COLOROP, D3DTOP_MODULATE );
    Direct3DDevice->SetTextureStageState( 0, D3DTSS_COLORARG1, D3DTA_TEXTURE );
    Direct3DDevice->SetTextureStageState( 0, D3DTSS_COLORARG2, D3DTA_DIFFUSE );
    Direct3DDevice->SetTextureStageState( 0, D3DTSS_ALPHAOP, D3DTOP_DISABLE );

    Direct3DDevice->SetTextureStageState( 0, D3DTSS_MAGFILTER, D3DTEXF_LINEAR );
    Direct3DDevice->SetTextureStageState( 0, D3DTSS_MINFILTER, D3DTEXF_LINEAR );
    Direct3DDevice->SetTextureStageState( 0, D3DTSS_MIPFILTER, D3DTEXF_NONE );

    Direct3DDevice->SetTexture( 1, MetaBalls_Texture2 );
    Direct3DDevice->SetTextureStageState( 1, D3DTSS_TEXTURETRANSFORMFLAGS, D3DTTFF_COUNT2 );
    Direct3DDevice->SetTextureStageState( 1, D3DTSS_TEXCOORDINDEX, D3DTSS_TCI_CAMERASPACENORMAL );

    Direct3DDevice->SetTextureStageState( 1, D3DTSS_COLOROP, D3DTOP_MODULATE2X );
    Direct3DDevice->SetTextureStageState( 1, D3DTSS_COLORARG1, D3DTA_TEXTURE );
    Direct3DDevice->SetTextureStageState( 1, D3DTSS_COLORARG2, D3DTA_CURRENT );
    Direct3DDevice->SetTextureStageState( 1, D3DTSS_ALPHAOP, D3DTOP_DISABLE );

    Direct3DDevice->SetTextureStageState( 1, D3DTSS_MAGFILTER, D3DTEXF_LINEAR );
    Direct3DDevice->SetTextureStageState( 1, D3DTSS_MINFILTER, D3DTEXF_LINEAR );
    Direct3DDevice->SetTextureStageState( 1, D3DTSS_MIPFILTER, D3DTEXF_NONE );

    Direct3DDevice->SetTextureStageState( 2, D3DTSS_COLOROP, D3DTOP_DISABLE );

    // --- Setup Material.
    memclear( &MetaBalls_Material1, sizeof( D3DMATERIAL8 ));
    MetaBalls_Material1.Diffuse.r = MetaBalls_Material1.Ambient.r = TimeDist;
    MetaBalls_Material1.Diffuse.g = MetaBalls_Material1.Ambient.g = TimeDist;
    MetaBalls_Material1.Diffuse.b = MetaBalls_Material1.Ambient.b = TimeDist;
    Direct3DDevice->SetMaterial( &MetaBalls_Material1 );

    Direct3DDevice->SetRenderState( D3DRS_AMBIENT, 0xFFFFFF );
    Direct3DDevice->SetRenderState( D3DRS_LIGHTING, TRUE );
    Direct3DDevice->LightEnable( 0, FALSE );

    // Direct3DDevice->SetRenderState( D3DRS_CULLMODE, D3DCULL_CCW );
    Direct3DDevice->SetRenderState( D3DRS_CULLMODE, D3DCULL_NONE );
    Direct3DDevice->SetRenderState( D3DRS_STENCILENABLE, FALSE );

    // --- Render.
    Direct3DDevice->SetVertexShader( D3DFVF_METABALLS_VERTEX1 );
    Direct3DDevice->SetStreamSource( 0, MetaBalls_VertexBuffer1, sizeof( METABALLS_VERTEX1 ));
    Direct3DDevice->DrawPrimitive( D3DPT_TRIANGLELIST, 0, MetaBalls_VertexIdx1/3 );
/*
    if ( MetaBalls_VertexIdx1/3 > MetaBalls_MaxPolygon )
      MetaBalls_MaxPolygon = MetaBalls_VertexIdx1/3;
//*/

    // === 2ND PASS.
//*
    MetaBalls_MarchingCubes( 3.0f * Fsqr( 1.0f - TimeDist ) + 0.2f + 0.01f * Fsin( Time * 0.1f ), 0 );

    D3DUMatrixTranslation( &D3DMatrix_Texture0, 0.5f * Fsin( Time * 0.06f ), 0.5f * Fcos( Time * 0.034f ), 0.5f * Fsin( Time * 0.047f ));
    D3DUMatrixRotationZ( &D3DMatrix_Temp, -Time * 0.01f );
    D3DUMatrixMultiply( &D3DMatrix_Texture0, &D3DMatrix_Texture0, &D3DMatrix_Temp );
    D3DUMatrixMultiply( &D3DMatrix_Texture0, &D3DMatrix_Texture0, &MetaBalls_TextureMatrix0 );
    Direct3DDevice->SetTransform( D3DTS_TEXTURE0, &D3DMatrix_Texture0 );
    Direct3DDevice->SetTransform( D3DTS_TEXTURE1, &MetaBalls_TextureMatrix0 );

    Direct3DDevice->SetTexture( 0, MetaBalls_Texture3 );
    Direct3DDevice->SetTextureStageState( 0, D3DTSS_TEXTURETRANSFORMFLAGS, D3DTTFF_COUNT2 );
    Direct3DDevice->SetTextureStageState( 0, D3DTSS_TEXCOORDINDEX, D3DTSS_TCI_CAMERASPACENORMAL );

    Direct3DDevice->SetTextureStageState( 0, D3DTSS_COLOROP, D3DTOP_MODULATE );
    Direct3DDevice->SetTextureStageState( 0, D3DTSS_COLORARG1, D3DTA_TEXTURE );
    Direct3DDevice->SetTextureStageState( 0, D3DTSS_COLORARG2, D3DTA_DIFFUSE );
    Direct3DDevice->SetTextureStageState( 0, D3DTSS_ALPHAOP, D3DTOP_DISABLE );

    Direct3DDevice->SetTextureStageState( 0, D3DTSS_MAGFILTER, D3DTEXF_LINEAR );
    Direct3DDevice->SetTextureStageState( 0, D3DTSS_MINFILTER, D3DTEXF_LINEAR );
    Direct3DDevice->SetTextureStageState( 0, D3DTSS_MIPFILTER, D3DTEXF_NONE );

    Direct3DDevice->SetTexture( 1, MetaBalls_Texture2 );
    Direct3DDevice->SetTextureStageState( 1, D3DTSS_TEXTURETRANSFORMFLAGS, D3DTTFF_COUNT2 );
    Direct3DDevice->SetTextureStageState( 1, D3DTSS_TEXCOORDINDEX, D3DTSS_TCI_CAMERASPACENORMAL );

    Direct3DDevice->SetTextureStageState( 1, D3DTSS_COLOROP, D3DTOP_MODULATE );
    Direct3DDevice->SetTextureStageState( 1, D3DTSS_COLORARG1, D3DTA_TEXTURE );
    Direct3DDevice->SetTextureStageState( 1, D3DTSS_COLORARG2, D3DTA_CURRENT );
    Direct3DDevice->SetTextureStageState( 1, D3DTSS_ALPHAOP, D3DTOP_DISABLE );

    Direct3DDevice->SetTextureStageState( 1, D3DTSS_MAGFILTER, D3DTEXF_LINEAR );
    Direct3DDevice->SetTextureStageState( 1, D3DTSS_MINFILTER, D3DTEXF_LINEAR );
    Direct3DDevice->SetTextureStageState( 1, D3DTSS_MIPFILTER, D3DTEXF_NONE );
  
    Direct3DDevice->SetTextureStageState( 2, D3DTSS_COLOROP, D3DTOP_DISABLE );

//  Direct3DDevice->SetRenderState( D3DRS_TEXTUREFACTOR, 0x010101 * Fround( 255.0f * TimeDist ));

    // --- Blending.
    Direct3DDevice->SetRenderState( D3DRS_SRCBLEND, D3DBLEND_ONE );
    Direct3DDevice->SetRenderState( D3DRS_DESTBLEND, D3DBLEND_ONE );
    Direct3DDevice->SetRenderState( D3DRS_ALPHABLENDENABLE, TRUE );

//  Direct3DDevice->SetRenderState( D3DRS_ZENABLE, FALSE );

    // --- Render.
    Direct3DDevice->SetVertexShader( D3DFVF_METABALLS_VERTEX1 );
    Direct3DDevice->SetStreamSource( 0, MetaBalls_VertexBuffer1, sizeof( METABALLS_VERTEX1 ));
    Direct3DDevice->DrawPrimitive( D3DPT_TRIANGLELIST, 0, MetaBalls_VertexIdx1/3 );
//*/
    Direct3DDevice->SetRenderState( D3DRS_ALPHABLENDENABLE, FALSE );
    Direct3DDevice->SetTextureStageState( 0, D3DTSS_TEXCOORDINDEX, D3DTSS_TCI_PASSTHRU );
    Direct3DDevice->SetTextureStageState( 1, D3DTSS_TEXCOORDINDEX, D3DTSS_TCI_PASSTHRU );
   }

// ------------------------------------------------------------------------------------------------

void MetaBalls_Render2( float Time )
  {
    int I, J, K, MB;
    float Dist, TimeDist;

    METABALLS = 6;

    MetaBalls_ActFrame++;

    MetaBalls_Max = 7.0f;
    MetaBalls_Max1 = Fsqr( MetaBalls_Max );
    MetaBalls_Max2 = 1.0f / MetaBalls_Max1;

    for ( I = 0; I < GRID; I++ )
      for ( J = 0; J < GRID; J++ )
        for ( K = 0; K < GRID; K++ )
          MetaBalls_Field[ I*GRIDS + J*GRID + K ] = 0.0f;

    TimeDist = 0.0f;
    if (( Time > 17.0f * 64.0f - 16.0f ) && ( Time <= 17.0f * 64.0 + 16.0f )) TimeDist = ( Time - ( 17.0f * 64.0f - 16.0f )) / 32.0f;
    if (( Time > 17.0f * 64.0f + 16.0f ) && ( Time <= 19.0f * 64.0f - 32.0f )) TimeDist = 1.0f;
    if (( Time > 19.0f * 64.0f - 32.0f ) && ( Time <= 19.0f * 64.0f )) TimeDist = ( 19.0f * 64.0f - Time ) / 32.0f;
    if ( Time > 19.0f *64.0f ) TimeDist = 0.0f;

    for ( MB = 0; MB < METABALLS; MB++ )
      {
        D3DUVec3Create( &MetaBalls_Position[ MB ],
          GRID2 + TimeDist * ( 5.0f + 2.0f * Fsin( Time * (float) ( MB+5 ) * 0.028f )) * Fcos(( (float) MB / 8.0f ) + Time * (float) ( MB+3 ) * 0.012f ),
          GRID2 + TimeDist * ( 5.0f + 2.0f * Fsin( Time * (float) ( MB+3 ) * 0.026f )) * Fsin(( (float) MB / 8.0f ) + Time * (float) ( MB+4 ) * 0.009f ),
          GRID2 + TimeDist * 6.0f * Fcos(( (float) MB / 8.0f ) + Time * (float) ( MB+5 ) * 0.011f ));
          
        for ( I = 0; I < GRID; I++ )
          for ( J = 0; J < GRID; J++ )
            for ( K = 0; K < GRID; K++ )
              {
                Dist = Fsqr( I - MetaBalls_Position[ MB ].x ) + Fsqr( J - MetaBalls_Position[ MB ].y ) + Fsqr( K - MetaBalls_Position[ MB ].z );
                if ( Dist < MetaBalls_Max1 )
                  {
                    if ( MB & 1 )
                      MetaBalls_Field[ I*GRIDS + J*GRID + K ] -= Fsqr( 1.0f - Dist * MetaBalls_Max2 ); else
                      MetaBalls_Field[ I*GRIDS + J*GRID + K ] += Fsqr( 1.0f - Dist * MetaBalls_Max2 );
                    MetaBalls_Frame[ I*GRIDS + J*GRID + K ] = MetaBalls_ActFrame;
                   }
               }
       }

    MetaBalls_MarchingCubes( 3.0f * Fsqr( 1.0f - TimeDist ) + 0.2f + 0.03f * Fsin( Time * 0.02f ), 0 /* METABALLS_NEGATIVE */ );

    Direct3DDevice->SetRenderState( D3DRS_ALPHABLENDENABLE, FALSE );

    // --- Setup World Matrix.
    Direct3DDevice->SetTransform( D3DTS_WORLD, &D3DUMatrixIdentity );

    // --- Setup View Matrix.
    D3DUVec3Create( &D3DVector_CameraPosition, 16.0f, 16.0f, -10.0f );
    D3DUVec3Create( &D3DVector_CameraTarget, -13.0f, -6.0f, 0.0f );
    D3DUVec3Create( &D3DVector_Up, 0.0f, -1.0f, 0.0f );
    D3DUMatrixLookAtLH( &D3DMatrix_View, &D3DVector_CameraPosition, &D3DVector_CameraTarget, &D3DVector_Up );
    Direct3DDevice->SetTransform( D3DTS_VIEW, &D3DMatrix_View );

    // --- Setup Projection Matrix.
    D3DUMatrixPerspectiveFovLH( &D3DMatrix_Projection, D3DU_PI * 0.4f, 1.778f, 0.1f, 10000.0f );
    Direct3DDevice->SetTransform( D3DTS_PROJECTION, &D3DMatrix_Projection );

    // --- Setup Texture Matrix.
    D3DUMatrixTranslation( &D3DMatrix_Texture0, 0.3f * Fsin( Time * 0.006f ), 0.3f * Fcos( Time * 0.0034f ), 0.3f * Fsin( Time * 0.0047f ));
    D3DUMatrixRotationZ( &D3DMatrix_Temp, Time * 0.01f );
    D3DUMatrixMultiply( &D3DMatrix_Texture0, &D3DMatrix_Texture0, &D3DMatrix_Temp );
    D3DUMatrixMultiply( &D3DMatrix_Texture0, &D3DMatrix_Texture0, &MetaBalls_TextureMatrix0 );
    Direct3DDevice->SetTransform( D3DTS_TEXTURE0, &D3DMatrix_Texture0 );
    Direct3DDevice->SetTransform( D3DTS_TEXTURE1, &MetaBalls_TextureMatrix0 );

    // --- Setup Textures.
    Direct3DDevice->SetTexture( 0, MetaBalls_Texture1 );
    Direct3DDevice->SetTextureStageState( 0, D3DTSS_TEXTURETRANSFORMFLAGS, D3DTTFF_COUNT2 );
    Direct3DDevice->SetTextureStageState( 0, D3DTSS_TEXCOORDINDEX, D3DTSS_TCI_CAMERASPACENORMAL );

    Direct3DDevice->SetTextureStageState( 0, D3DTSS_COLOROP, D3DTOP_MODULATE );
    Direct3DDevice->SetTextureStageState( 0, D3DTSS_COLORARG1, D3DTA_TEXTURE );
    Direct3DDevice->SetTextureStageState( 0, D3DTSS_COLORARG2, D3DTA_CURRENT );
    Direct3DDevice->SetTextureStageState( 0, D3DTSS_ALPHAOP, D3DTOP_DISABLE );

    Direct3DDevice->SetTextureStageState( 0, D3DTSS_MAGFILTER, D3DTEXF_LINEAR );
    Direct3DDevice->SetTextureStageState( 0, D3DTSS_MINFILTER, D3DTEXF_LINEAR );
    Direct3DDevice->SetTextureStageState( 0, D3DTSS_MIPFILTER, D3DTEXF_NONE );

    Direct3DDevice->SetTexture( 1, MetaBalls_Texture2 );
    Direct3DDevice->SetTextureStageState( 1, D3DTSS_TEXTURETRANSFORMFLAGS, D3DTTFF_COUNT2 );
    Direct3DDevice->SetTextureStageState( 1, D3DTSS_TEXCOORDINDEX, D3DTSS_TCI_CAMERASPACENORMAL );

    Direct3DDevice->SetTextureStageState( 1, D3DTSS_COLOROP, D3DTOP_MODULATE2X );
    Direct3DDevice->SetTextureStageState( 1, D3DTSS_COLORARG1, D3DTA_TEXTURE );
    Direct3DDevice->SetTextureStageState( 1, D3DTSS_COLORARG2, D3DTA_CURRENT );
    Direct3DDevice->SetTextureStageState( 1, D3DTSS_ALPHAOP, D3DTOP_DISABLE );

    Direct3DDevice->SetTextureStageState( 1, D3DTSS_MAGFILTER, D3DTEXF_LINEAR );
    Direct3DDevice->SetTextureStageState( 1, D3DTSS_MINFILTER, D3DTEXF_LINEAR );
    Direct3DDevice->SetTextureStageState( 1, D3DTSS_MIPFILTER, D3DTEXF_NONE );

    Direct3DDevice->SetTextureStageState( 2, D3DTSS_COLOROP, D3DTOP_DISABLE );

    // --- Setup Material.
    memclear( &MetaBalls_Material1, sizeof( D3DMATERIAL8 ));
    MetaBalls_Material1.Diffuse.r = MetaBalls_Material1.Ambient.r = TimeDist;
    MetaBalls_Material1.Diffuse.g = MetaBalls_Material1.Ambient.g = TimeDist;
    MetaBalls_Material1.Diffuse.b = MetaBalls_Material1.Ambient.b = TimeDist;
    Direct3DDevice->SetMaterial( &MetaBalls_Material1 );

    Direct3DDevice->SetRenderState( D3DRS_AMBIENT, 0xFFFFFF );
    Direct3DDevice->SetRenderState( D3DRS_LIGHTING, TRUE );
    Direct3DDevice->LightEnable( 0, FALSE );

    Direct3DDevice->SetRenderState( D3DRS_CULLMODE, D3DCULL_NONE );
    Direct3DDevice->SetRenderState( D3DRS_STENCILENABLE, FALSE );

    // --- Render.
    Direct3DDevice->SetVertexShader( D3DFVF_METABALLS_VERTEX1 );
    Direct3DDevice->SetStreamSource( 0, MetaBalls_VertexBuffer1, sizeof( METABALLS_VERTEX1 ));
    Direct3DDevice->DrawPrimitive( D3DPT_TRIANGLELIST, 0, MetaBalls_VertexIdx1/3 );

    // === 2ND PASS.

    MetaBalls_MarchingCubes( 3.0f * Fsqr( 1.0f - TimeDist ) + 0.2f + 0.03f * Fsin( Time * 0.03f ), METABALLS_NEGATIVE );

    // --- Setup Texture Matrix.
    D3DUMatrixTranslation( &D3DMatrix_Texture0, 0.3f * Fsin( Time * 0.006f ), 0.3f * Fcos( Time * 0.0034f ), 0.3f * Fsin( Time * 0.0047f ));
    D3DUMatrixRotationZ( &D3DMatrix_Temp, Time * 0.01f );
    D3DUMatrixMultiply( &D3DMatrix_Texture0, &D3DMatrix_Texture0, &D3DMatrix_Temp );
    D3DUMatrixMultiply( &D3DMatrix_Texture0, &D3DMatrix_Texture0, &MetaBalls_TextureMatrix0 );
    Direct3DDevice->SetTransform( D3DTS_TEXTURE0, &D3DMatrix_Texture0 );
    Direct3DDevice->SetTransform( D3DTS_TEXTURE1, &MetaBalls_TextureMatrix0 );

    // --- Setup Textures.
    Direct3DDevice->SetTexture( 0, MetaBalls_Texture4 );
    Direct3DDevice->SetTextureStageState( 0, D3DTSS_TEXTURETRANSFORMFLAGS, D3DTTFF_COUNT2 );
    Direct3DDevice->SetTextureStageState( 0, D3DTSS_TEXCOORDINDEX, D3DTSS_TCI_CAMERASPACENORMAL );

    Direct3DDevice->SetTextureStageState( 0, D3DTSS_COLOROP, D3DTOP_MODULATE );
    Direct3DDevice->SetTextureStageState( 0, D3DTSS_COLORARG1, D3DTA_TEXTURE );
    Direct3DDevice->SetTextureStageState( 0, D3DTSS_COLORARG2, D3DTA_DIFFUSE );
    Direct3DDevice->SetTextureStageState( 0, D3DTSS_ALPHAOP, D3DTOP_DISABLE );

    Direct3DDevice->SetTextureStageState( 0, D3DTSS_MAGFILTER, D3DTEXF_LINEAR );
    Direct3DDevice->SetTextureStageState( 0, D3DTSS_MINFILTER, D3DTEXF_LINEAR );
    Direct3DDevice->SetTextureStageState( 0, D3DTSS_MIPFILTER, D3DTEXF_NONE );

    Direct3DDevice->SetTexture( 1, MetaBalls_Texture2 );
    Direct3DDevice->SetTextureStageState( 1, D3DTSS_TEXTURETRANSFORMFLAGS, D3DTTFF_COUNT2 );
    Direct3DDevice->SetTextureStageState( 1, D3DTSS_TEXCOORDINDEX, D3DTSS_TCI_CAMERASPACENORMAL );

    Direct3DDevice->SetTextureStageState( 1, D3DTSS_COLOROP, D3DTOP_MODULATE2X );
    Direct3DDevice->SetTextureStageState( 1, D3DTSS_COLORARG1, D3DTA_TEXTURE );
    Direct3DDevice->SetTextureStageState( 1, D3DTSS_COLORARG2, D3DTA_CURRENT );
    Direct3DDevice->SetTextureStageState( 1, D3DTSS_ALPHAOP, D3DTOP_DISABLE );

    Direct3DDevice->SetTextureStageState( 1, D3DTSS_MAGFILTER, D3DTEXF_LINEAR );
    Direct3DDevice->SetTextureStageState( 1, D3DTSS_MINFILTER, D3DTEXF_LINEAR );
    Direct3DDevice->SetTextureStageState( 1, D3DTSS_MIPFILTER, D3DTEXF_NONE );

    Direct3DDevice->SetTextureStageState( 2, D3DTSS_COLOROP, D3DTOP_DISABLE );

    // --- Render.
    Direct3DDevice->SetVertexShader( D3DFVF_METABALLS_VERTEX1 );
    Direct3DDevice->SetStreamSource( 0, MetaBalls_VertexBuffer1, sizeof( METABALLS_VERTEX1 ));
    Direct3DDevice->DrawPrimitive( D3DPT_TRIANGLELIST, 0, MetaBalls_VertexIdx1/3 );

    // === Render Transparent Shell.

    MetaBalls_Max = 8.0f;
    MetaBalls_Max1 = Fsqr( MetaBalls_Max );
    MetaBalls_Max2 = 1.0f / MetaBalls_Max1;

    for ( I = 0; I < GRID; I++ )
      for ( J = 0; J < GRID; J++ )
        for ( K = 0; K < GRID; K++ )
          MetaBalls_Field[ I*GRIDS + J*GRID + K ] = 0.0f;

    for ( MB = 0; MB < METABALLS; MB++ )
      {
        for ( I = 0; I < GRID; I++ )
          for ( J = 0; J < GRID; J++ )
            for ( K = 0; K < GRID; K++ )
              {
                Dist = Fsqr( I - MetaBalls_Position[ MB ].x ) + Fsqr( J - MetaBalls_Position[ MB ].y ) + Fsqr( K - MetaBalls_Position[ MB ].z );
                if ( Dist < MetaBalls_Max1 )
                  {
                    MetaBalls_Field[ I*GRIDS + J*GRID + K ] += Fsqr( 1.0f - Dist * MetaBalls_Max2 );
                    MetaBalls_Frame[ I*GRIDS + J*GRID + K ] = MetaBalls_ActFrame;
                   }
               }
       }

    MetaBalls_MarchingCubes( 3.0f * Fsqr( 1.0f - TimeDist ) + 0.2f + 0.01f * Fsin( Time * 0.013f ), 0 );

    D3DUMatrixTranslation( &D3DMatrix_Texture0, 0.5f * Fsin( Time * 0.006f ), 0.5f * Fcos( Time * 0.0034f ), 0.5f * Fsin( Time * 0.0047f ));
    D3DUMatrixRotationZ( &D3DMatrix_Temp, -Time * 0.01f );
    D3DUMatrixMultiply( &D3DMatrix_Texture0, &D3DMatrix_Texture0, &D3DMatrix_Temp );
    D3DUMatrixMultiply( &D3DMatrix_Texture0, &D3DMatrix_Texture0, &MetaBalls_TextureMatrix0 );
    Direct3DDevice->SetTransform( D3DTS_TEXTURE0, &D3DMatrix_Texture0 );
    Direct3DDevice->SetTransform( D3DTS_TEXTURE1, &MetaBalls_TextureMatrix0 );

    // --- Setup Textures.
    Direct3DDevice->SetTexture( 0, MetaBalls_Texture3 );
    Direct3DDevice->SetTextureStageState( 0, D3DTSS_TEXTURETRANSFORMFLAGS, D3DTTFF_COUNT2 );
    Direct3DDevice->SetTextureStageState( 0, D3DTSS_TEXCOORDINDEX, D3DTSS_TCI_CAMERASPACENORMAL );

    Direct3DDevice->SetTextureStageState( 0, D3DTSS_COLOROP, D3DTOP_MODULATE );
    Direct3DDevice->SetTextureStageState( 0, D3DTSS_COLORARG1, D3DTA_TEXTURE );
    Direct3DDevice->SetTextureStageState( 0, D3DTSS_COLORARG2, D3DTA_DIFFUSE );
    Direct3DDevice->SetTextureStageState( 0, D3DTSS_ALPHAOP, D3DTOP_DISABLE );

    Direct3DDevice->SetTextureStageState( 0, D3DTSS_MAGFILTER, D3DTEXF_LINEAR );
    Direct3DDevice->SetTextureStageState( 0, D3DTSS_MINFILTER, D3DTEXF_LINEAR );
    Direct3DDevice->SetTextureStageState( 0, D3DTSS_MIPFILTER, D3DTEXF_NONE );

    Direct3DDevice->SetTexture( 1, MetaBalls_Texture2 );
    Direct3DDevice->SetTextureStageState( 1, D3DTSS_TEXTURETRANSFORMFLAGS, D3DTTFF_COUNT2 );
    Direct3DDevice->SetTextureStageState( 1, D3DTSS_TEXCOORDINDEX, D3DTSS_TCI_CAMERASPACENORMAL );

    Direct3DDevice->SetTextureStageState( 1, D3DTSS_COLOROP, D3DTOP_MODULATE );
    Direct3DDevice->SetTextureStageState( 1, D3DTSS_COLORARG1, D3DTA_TEXTURE );
    Direct3DDevice->SetTextureStageState( 1, D3DTSS_COLORARG2, D3DTA_CURRENT );
    Direct3DDevice->SetTextureStageState( 1, D3DTSS_ALPHAOP, D3DTOP_DISABLE );

    Direct3DDevice->SetTextureStageState( 1, D3DTSS_MAGFILTER, D3DTEXF_LINEAR );
    Direct3DDevice->SetTextureStageState( 1, D3DTSS_MINFILTER, D3DTEXF_LINEAR );
    Direct3DDevice->SetTextureStageState( 1, D3DTSS_MIPFILTER, D3DTEXF_NONE );

    Direct3DDevice->SetTextureStageState( 2, D3DTSS_COLOROP, D3DTOP_DISABLE );

    // --- Blending.
    Direct3DDevice->SetRenderState( D3DRS_SRCBLEND, D3DBLEND_ONE );
    Direct3DDevice->SetRenderState( D3DRS_DESTBLEND, D3DBLEND_ONE );
    Direct3DDevice->SetRenderState( D3DRS_ALPHABLENDENABLE, TRUE );

    // --- Render.
    Direct3DDevice->SetVertexShader( D3DFVF_METABALLS_VERTEX1 );
    Direct3DDevice->SetStreamSource( 0, MetaBalls_VertexBuffer1, sizeof( METABALLS_VERTEX1 ));
    Direct3DDevice->DrawPrimitive( D3DPT_TRIANGLELIST, 0, MetaBalls_VertexIdx1/3 );

    Direct3DDevice->SetRenderState( D3DRS_ALPHABLENDENABLE, FALSE );
    Direct3DDevice->SetTextureStageState( 0, D3DTSS_TEXCOORDINDEX, D3DTSS_TCI_PASSTHRU );
    Direct3DDevice->SetTextureStageState( 1, D3DTSS_TEXCOORDINDEX, D3DTSS_TCI_PASSTHRU );
   }

// ------------------------------------------------------------------------------------------------

void MetaBalls_Render3( float Time )
  {
    int I, J, K, MB;
    float X0, Y0, Z0, DB, Dist, TimeDist;

    METABALLS = 3;

    MetaBalls_ActFrame++;

    TimeDist = 0.0f;
    if (( Time > 21.0f * 64.0f ) && ( Time <= 21.0f * 64.0 + 32.0f )) TimeDist = ( Time - ( 21.0f * 64.0f )) / 32.0f;
    if (( Time > 21.0f * 64.0f + 32.0f ) && ( Time <= 25.0f * 64.0f - 32.0f )) TimeDist = 1.0f;
    if (( Time > 25.0f * 64.0f - 32.0f ) && ( Time <= 25.0f * 64.0f )) TimeDist = ( 25.0f * 64.0f - Time ) / 32.0f;
    if ( Time > 25.0f *64.0f ) TimeDist = 0.0f;

    MetaBalls_Max = 5.5f * TimeDist;
    MetaBalls_Max1 = Fsqr( MetaBalls_Max );
    MetaBalls_Max2 = 1.0f / MetaBalls_Max1;

    float Fcos_Time007 = Fcos( Time*0.07f );
    float Fsin_Time007 = Fsin( Time*0.07f );

    for ( I = 0; I < GRID; I++ )
      for ( K = 0; K < GRID; K++ )
        {
          X0 = Fsqr( Fcos_Time007 * ( I-15.5f) - Fsin_Time007 * ( K-15.5f ));
          Z0 = Fsqr( Fsin_Time007 * ( I-15.5f) + Fcos_Time007 * ( K-15.5f ));
          for ( J = 0; J < GRID; J++ )
            {
              Y0 = Fsqr( J-15.5f );
              // DB = 9.0f - Fsqrt( X0*X0 + Y0*Y0 );
              Dist = Z0 + 81.0f - ( X0 + Y0 );
              if ( Dist < MetaBalls_Max1 ) 
                { 
                  MetaBalls_Field[ I*GRIDS + J*GRID + K ] = Fsqr( 1.0f - Dist * MetaBalls_Max2 );
                  MetaBalls_Frame[ I*GRIDS + J*GRID + K ] = MetaBalls_ActFrame;
                 } else
                  MetaBalls_Field[ I*GRIDS + J*GRID + K ] = 0.0f;
             }
        }

    MetaBalls_Max = 7.0f;
    MetaBalls_Max1 = Fsqr( MetaBalls_Max );
    MetaBalls_Max2 = 1.0f / MetaBalls_Max1;

    for ( MB = 0; MB < METABALLS; MB++ )
      {
        D3DUVec3Create( &MetaBalls_Position[ MB ],
          GRID2 + TimeDist * ( 5.0f + 2.0f * Fsin( Time * (float) ( MB+5 ) * 0.028f )) * Fcos(( (float) MB / 8.0f ) + Time * (float) ( MB+3 ) * 0.012f ),
          GRID2 + TimeDist * ( 5.0f + 2.0f * Fsin( Time * (float) ( MB+3 ) * 0.026f )) * Fsin(( (float) MB / 8.0f ) + Time * (float) ( MB+4 ) * 0.009f ),
          GRID2 + TimeDist * 6.0f * Fcos(( (float) MB / 8.0f ) + Time * (float) ( MB+5 ) * 0.011f ));
          
        for ( I = 0; I < GRID; I++ )
          for ( J = 0; J < GRID; J++ )
            for ( K = 0; K < GRID; K++ )
              {
                Dist = Fsqr( I - MetaBalls_Position[ MB ].x ) + Fsqr( J - MetaBalls_Position[ MB ].y ) + Fsqr( K - MetaBalls_Position[ MB ].z );
                if ( Dist < MetaBalls_Max1 )
                  {
//                  if ( MB & 1 )
                      MetaBalls_Field[ I*GRIDS + J*GRID + K ] -= Fsqr( 1.0f - Dist * MetaBalls_Max2 );// else
//                    MetaBalls_Field[ I*GRIDS + J*GRID + K ] += Fsqr( 1.0f - Dist * MetaBalls_Max2 );
                    MetaBalls_Frame[ I*GRIDS + J*GRID + K ] = MetaBalls_ActFrame;
                   }
               }
       }

    MetaBalls_MarchingCubes( 6.0f * Fsqr( 1.0f - TimeDist ) + 0.2f + 0.03f * Fsin( Time * 0.02f ), 0 /* METABALLS_NEGATIVE */ );

    Direct3DDevice->SetRenderState( D3DRS_ALPHABLENDENABLE, FALSE );

    // --- Setup World Matrix.
    Direct3DDevice->SetTransform( D3DTS_WORLD, &D3DUMatrixIdentity );

    // --- Setup View Matrix.
    D3DUVec3Create( &D3DVector_CameraPosition, 16.0f, 16.0f, -10.0f );
    D3DUVec3Create( &D3DVector_CameraTarget, -13.0f, -6.0f, 0.0f );
    D3DUVec3Create( &D3DVector_Up, 0.0f, -1.0f, 0.0f );
    D3DUMatrixLookAtLH( &D3DMatrix_View, &D3DVector_CameraPosition, &D3DVector_CameraTarget, &D3DVector_Up );
    Direct3DDevice->SetTransform( D3DTS_VIEW, &D3DMatrix_View );

    // --- Setup Projection Matrix.
    D3DUMatrixPerspectiveFovLH( &D3DMatrix_Projection, D3DU_PI * 0.4f, 1.778f, 0.1f, 10000.0f );
    Direct3DDevice->SetTransform( D3DTS_PROJECTION, &D3DMatrix_Projection );

    // --- Setup Texture Matrix.
    D3DUMatrixTranslation( &D3DMatrix_Texture0, 0.3f * Fsin( Time * 0.006f ), 0.3f * Fcos( Time * 0.0034f ), 0.3f * Fsin( Time * 0.0047f ));
    D3DUMatrixRotationZ( &D3DMatrix_Temp, Time * 0.01f );
    D3DUMatrixMultiply( &D3DMatrix_Texture0, &D3DMatrix_Texture0, &D3DMatrix_Temp );
    D3DUMatrixMultiply( &D3DMatrix_Texture0, &D3DMatrix_Texture0, &MetaBalls_TextureMatrix0 );
    Direct3DDevice->SetTransform( D3DTS_TEXTURE0, &D3DMatrix_Texture0 );
    Direct3DDevice->SetTransform( D3DTS_TEXTURE1, &MetaBalls_TextureMatrix0 );

    // --- Setup Textures.
    Direct3DDevice->SetTexture( 0, MetaBalls_Texture1 );
    Direct3DDevice->SetTextureStageState( 0, D3DTSS_TEXTURETRANSFORMFLAGS, D3DTTFF_COUNT2 );
    Direct3DDevice->SetTextureStageState( 0, D3DTSS_TEXCOORDINDEX, D3DTSS_TCI_CAMERASPACENORMAL );

    Direct3DDevice->SetTextureStageState( 0, D3DTSS_COLOROP, D3DTOP_MODULATE );
    Direct3DDevice->SetTextureStageState( 0, D3DTSS_COLORARG1, D3DTA_TEXTURE );
    Direct3DDevice->SetTextureStageState( 0, D3DTSS_COLORARG2, D3DTA_CURRENT );
    Direct3DDevice->SetTextureStageState( 0, D3DTSS_ALPHAOP, D3DTOP_DISABLE );

    Direct3DDevice->SetTextureStageState( 0, D3DTSS_MAGFILTER, D3DTEXF_LINEAR );
    Direct3DDevice->SetTextureStageState( 0, D3DTSS_MINFILTER, D3DTEXF_LINEAR );
    Direct3DDevice->SetTextureStageState( 0, D3DTSS_MIPFILTER, D3DTEXF_NONE );

    Direct3DDevice->SetTexture( 1, MetaBalls_Texture2 );
    Direct3DDevice->SetTextureStageState( 1, D3DTSS_TEXTURETRANSFORMFLAGS, D3DTTFF_COUNT2 );
    Direct3DDevice->SetTextureStageState( 1, D3DTSS_TEXCOORDINDEX, D3DTSS_TCI_CAMERASPACENORMAL );

    Direct3DDevice->SetTextureStageState( 1, D3DTSS_COLOROP, D3DTOP_MODULATE2X );
    Direct3DDevice->SetTextureStageState( 1, D3DTSS_COLORARG1, D3DTA_TEXTURE );
    Direct3DDevice->SetTextureStageState( 1, D3DTSS_COLORARG2, D3DTA_CURRENT );
    Direct3DDevice->SetTextureStageState( 1, D3DTSS_ALPHAOP, D3DTOP_DISABLE );

    Direct3DDevice->SetTextureStageState( 1, D3DTSS_MAGFILTER, D3DTEXF_LINEAR );
    Direct3DDevice->SetTextureStageState( 1, D3DTSS_MINFILTER, D3DTEXF_LINEAR );
    Direct3DDevice->SetTextureStageState( 1, D3DTSS_MIPFILTER, D3DTEXF_NONE );

    Direct3DDevice->SetTextureStageState( 2, D3DTSS_COLOROP, D3DTOP_DISABLE );

    // --- Setup Material.
    memclear( &MetaBalls_Material1, sizeof( D3DMATERIAL8 ));
    MetaBalls_Material1.Diffuse.r = MetaBalls_Material1.Ambient.r = TimeDist;
    MetaBalls_Material1.Diffuse.g = MetaBalls_Material1.Ambient.g = TimeDist;
    MetaBalls_Material1.Diffuse.b = MetaBalls_Material1.Ambient.b = TimeDist;
    Direct3DDevice->SetMaterial( &MetaBalls_Material1 );

    Direct3DDevice->SetRenderState( D3DRS_AMBIENT, 0xFFFFFF );
    Direct3DDevice->SetRenderState( D3DRS_LIGHTING, TRUE );
    Direct3DDevice->LightEnable( 0, FALSE );

    Direct3DDevice->SetRenderState( D3DRS_CULLMODE, D3DCULL_NONE );
    Direct3DDevice->SetRenderState( D3DRS_STENCILENABLE, FALSE );

    // --- Render.
    Direct3DDevice->SetVertexShader( D3DFVF_METABALLS_VERTEX1 );
    Direct3DDevice->SetStreamSource( 0, MetaBalls_VertexBuffer1, sizeof( METABALLS_VERTEX1 ));
    Direct3DDevice->DrawPrimitive( D3DPT_TRIANGLELIST, 0, MetaBalls_VertexIdx1/3 );

    // === 2ND PASS.

    MetaBalls_MarchingCubes( 6.0f * Fsqr( 1.0f - TimeDist ) + 0.2f + 0.03f * Fsin( Time * 0.03f ), METABALLS_NEGATIVE );

    // --- Setup Texture Matrix.
    D3DUMatrixTranslation( &D3DMatrix_Texture0, 0.3f * Fsin( Time * 0.006f ), 0.3f * Fcos( Time * 0.0034f ), 0.3f * Fsin( Time * 0.0047f ));
    D3DUMatrixRotationZ( &D3DMatrix_Temp, Time * 0.01f );
    D3DUMatrixMultiply( &D3DMatrix_Texture0, &D3DMatrix_Texture0, &D3DMatrix_Temp );
    D3DUMatrixMultiply( &D3DMatrix_Texture0, &D3DMatrix_Texture0, &MetaBalls_TextureMatrix0 );
    Direct3DDevice->SetTransform( D3DTS_TEXTURE0, &D3DMatrix_Texture0 );
    Direct3DDevice->SetTransform( D3DTS_TEXTURE1, &MetaBalls_TextureMatrix0 );

    // --- Setup Textures.
    Direct3DDevice->SetTexture( 0, MetaBalls_Texture4 );
    Direct3DDevice->SetTextureStageState( 0, D3DTSS_TEXTURETRANSFORMFLAGS, D3DTTFF_COUNT2 );
    Direct3DDevice->SetTextureStageState( 0, D3DTSS_TEXCOORDINDEX, D3DTSS_TCI_CAMERASPACENORMAL );

    Direct3DDevice->SetTextureStageState( 0, D3DTSS_COLOROP, D3DTOP_MODULATE );
    Direct3DDevice->SetTextureStageState( 0, D3DTSS_COLORARG1, D3DTA_TEXTURE );
    Direct3DDevice->SetTextureStageState( 0, D3DTSS_COLORARG2, D3DTA_DIFFUSE );
    Direct3DDevice->SetTextureStageState( 0, D3DTSS_ALPHAOP, D3DTOP_DISABLE );

    Direct3DDevice->SetTextureStageState( 0, D3DTSS_MAGFILTER, D3DTEXF_LINEAR );
    Direct3DDevice->SetTextureStageState( 0, D3DTSS_MINFILTER, D3DTEXF_LINEAR );
    Direct3DDevice->SetTextureStageState( 0, D3DTSS_MIPFILTER, D3DTEXF_NONE );

    Direct3DDevice->SetTexture( 1, MetaBalls_Texture2 );
    Direct3DDevice->SetTextureStageState( 1, D3DTSS_TEXTURETRANSFORMFLAGS, D3DTTFF_COUNT2 );
    Direct3DDevice->SetTextureStageState( 1, D3DTSS_TEXCOORDINDEX, D3DTSS_TCI_CAMERASPACENORMAL );

    Direct3DDevice->SetTextureStageState( 1, D3DTSS_COLOROP, D3DTOP_MODULATE2X );
    Direct3DDevice->SetTextureStageState( 1, D3DTSS_COLORARG1, D3DTA_TEXTURE );
    Direct3DDevice->SetTextureStageState( 1, D3DTSS_COLORARG2, D3DTA_CURRENT );
    Direct3DDevice->SetTextureStageState( 1, D3DTSS_ALPHAOP, D3DTOP_DISABLE );

    Direct3DDevice->SetTextureStageState( 1, D3DTSS_MAGFILTER, D3DTEXF_LINEAR );
    Direct3DDevice->SetTextureStageState( 1, D3DTSS_MINFILTER, D3DTEXF_LINEAR );
    Direct3DDevice->SetTextureStageState( 1, D3DTSS_MIPFILTER, D3DTEXF_NONE );

    Direct3DDevice->SetTextureStageState( 2, D3DTSS_COLOROP, D3DTOP_DISABLE );

    // --- Render.
    Direct3DDevice->SetVertexShader( D3DFVF_METABALLS_VERTEX1 );
    Direct3DDevice->SetStreamSource( 0, MetaBalls_VertexBuffer1, sizeof( METABALLS_VERTEX1 ));
    Direct3DDevice->DrawPrimitive( D3DPT_TRIANGLELIST, 0, MetaBalls_VertexIdx1/3 );

    // === Render Transparent Shell.
/*
    MetaBalls_Max = 7.0f;
    MetaBalls_Max1 = Fsqr( MetaBalls_Max );
    MetaBalls_Max2 = 1.0f / MetaBalls_Max1;

    for ( I = 0; I < GRID; I++ )
      for ( J = 0; J < GRID; J++ )
         for ( K = 0; K < GRID; K++ )
          {
            X0 = Fcos( Time*0.07f ) * ( I-15.5f) - Fsin( Time*0.07f ) * ( K-15.5f );
            Y0 = J-15.5f;
            Z0 = Fsin( Time*0.07f ) * ( I-15.5f) + Fcos( Time*0.07f ) * ( K-15.5f );
            DB = 9.0f - Fsqrt( X0*X0 + Y0*Y0 );
            Dist = Z0*Z0 + DB*DB;
            if ( Dist < MetaBalls_Max1 ) 
              { 
                MetaBalls_Field[ I*GRIDS + J*GRID + K ] = Fsqr( 1.0f - Dist * MetaBalls_Max2 );
                MetaBalls_Frame[ I*GRIDS + J*GRID + K ] = MetaBalls_ActFrame;
               } else
                MetaBalls_Field[ I*GRIDS + J*GRID + K ] = 0.0f;
           }

    MetaBalls_Max = 8.0f;
    MetaBalls_Max1 = Fsqr( MetaBalls_Max );
    MetaBalls_Max2 = 1.0f / MetaBalls_Max1;

    for ( MB = 0; MB < METABALLS; MB++ )
      {
        for ( I = 0; I < GRID; I++ )
          for ( J = 0; J < GRID; J++ )
            for ( K = 0; K < GRID; K++ )
              {
                Dist = Fsqr( I - MetaBalls_Position[ MB ].x ) + Fsqr( J - MetaBalls_Position[ MB ].y ) + Fsqr( K - MetaBalls_Position[ MB ].z );
                if ( Dist < MetaBalls_Max1 )
                  {
                    MetaBalls_Field[ I*GRIDS + J*GRID + K ] += Fsqr( 1.0f - Dist * MetaBalls_Max2 );
                    MetaBalls_Frame[ I*GRIDS + J*GRID + K ] = MetaBalls_ActFrame;
                   }
               }
       }

    MetaBalls_MarchingCubes( 3.0f * Fsqr( 1.0f - TimeDist ) + 0.2f + 0.01f * Fsin( Time * 0.013f ), 0 );

    D3DUMatrixTranslation( &D3DMatrix_Texture0, 0.5f * Fsin( Time * 0.006f ), 0.5f * Fcos( Time * 0.0034f ), 0.5f * Fsin( Time * 0.0047f ));
    D3DUMatrixRotationZ( &D3DMatrix_Temp, -Time * 0.01f );
    D3DUMatrixMultiply( &D3DMatrix_Texture0, &D3DMatrix_Texture0, &D3DMatrix_Temp );
    D3DUMatrixMultiply( &D3DMatrix_Texture0, &D3DMatrix_Texture0, &MetaBalls_TextureMatrix0 );
    Direct3DDevice->SetTransform( D3DTS_TEXTURE0, &D3DMatrix_Texture0 );
    Direct3DDevice->SetTransform( D3DTS_TEXTURE1, &MetaBalls_TextureMatrix0 );

    // --- Setup Textures.
    Direct3DDevice->SetTexture( 0, MetaBalls_Texture3 );
    Direct3DDevice->SetTextureStageState( 0, D3DTSS_TEXTURETRANSFORMFLAGS, D3DTTFF_COUNT2 );
    Direct3DDevice->SetTextureStageState( 0, D3DTSS_TEXCOORDINDEX, D3DTSS_TCI_CAMERASPACENORMAL );

    Direct3DDevice->SetTextureStageState( 0, D3DTSS_COLOROP, D3DTOP_MODULATE );
    Direct3DDevice->SetTextureStageState( 0, D3DTSS_COLORARG1, D3DTA_TEXTURE );
    Direct3DDevice->SetTextureStageState( 0, D3DTSS_COLORARG2, D3DTA_DIFFUSE );
    Direct3DDevice->SetTextureStageState( 0, D3DTSS_ALPHAOP, D3DTOP_DISABLE );

    Direct3DDevice->SetTextureStageState( 0, D3DTSS_MAGFILTER, D3DTEXF_LINEAR );
    Direct3DDevice->SetTextureStageState( 0, D3DTSS_MINFILTER, D3DTEXF_LINEAR );
    Direct3DDevice->SetTextureStageState( 0, D3DTSS_MIPFILTER, D3DTEXF_NONE );

    Direct3DDevice->SetTexture( 1, MetaBalls_Texture2 );
    Direct3DDevice->SetTextureStageState( 1, D3DTSS_TEXTURETRANSFORMFLAGS, D3DTTFF_COUNT2 );
    Direct3DDevice->SetTextureStageState( 1, D3DTSS_TEXCOORDINDEX, D3DTSS_TCI_CAMERASPACENORMAL );

    Direct3DDevice->SetTextureStageState( 1, D3DTSS_COLOROP, D3DTOP_MODULATE );
    Direct3DDevice->SetTextureStageState( 1, D3DTSS_COLORARG1, D3DTA_TEXTURE );
    Direct3DDevice->SetTextureStageState( 1, D3DTSS_COLORARG2, D3DTA_CURRENT );
    Direct3DDevice->SetTextureStageState( 1, D3DTSS_ALPHAOP, D3DTOP_DISABLE );

    Direct3DDevice->SetTextureStageState( 1, D3DTSS_MAGFILTER, D3DTEXF_LINEAR );
    Direct3DDevice->SetTextureStageState( 1, D3DTSS_MINFILTER, D3DTEXF_LINEAR );
    Direct3DDevice->SetTextureStageState( 1, D3DTSS_MIPFILTER, D3DTEXF_NONE );

    Direct3DDevice->SetTextureStageState( 2, D3DTSS_COLOROP, D3DTOP_DISABLE );

    // --- Blending.
    Direct3DDevice->SetRenderState( D3DRS_SRCBLEND, D3DBLEND_ONE );
    Direct3DDevice->SetRenderState( D3DRS_DESTBLEND, D3DBLEND_ONE );
    Direct3DDevice->SetRenderState( D3DRS_ALPHABLENDENABLE, TRUE );

    // --- Render.
    Direct3DDevice->SetVertexShader( D3DFVF_METABALLS_VERTEX1 );
    Direct3DDevice->SetStreamSource( 0, MetaBalls_VertexBuffer1, sizeof( METABALLS_VERTEX1 ));
    Direct3DDevice->DrawPrimitive( D3DPT_TRIANGLELIST, 0, MetaBalls_VertexIdx1/3 );
*/
    Direct3DDevice->SetRenderState( D3DRS_ALPHABLENDENABLE, FALSE );
    Direct3DDevice->SetTextureStageState( 0, D3DTSS_TEXCOORDINDEX, D3DTSS_TCI_PASSTHRU );
    Direct3DDevice->SetTextureStageState( 1, D3DTSS_TEXCOORDINDEX, D3DTSS_TCI_PASSTHRU );
  }

