org 100h

; ---------------------------------------------------------------------------
; ❄️ THE VCCC2025 SNOWRUNE QUEST ❄️ (a tiny Christmas demoscene tale) 
;
; In a quiet DOS winter night, a small COM elf wakes up at 0100h.
; No sprites. No pixels. Only text cells, bits, and a stolen sleigh ride.
;
; Legend says a snowflake can be summoned from just 10 runes (16-bit masks):
; read them forward to climb the sky, then rewind the spellbook to mirror
; the second half. The rest is pure trickery: VRAM theft, perfect strides,
; and one '*' that twinkles when a bit says "yes".
;
; Your mission:
;   1) Open the North Pole portal (B800h).
;   2) Speak the runes, bit by bit, into a 16-cell chant.
;   3) Let an elf clone 3 ornaments to fatten the arms.
;   4) Ride the sleigh: 38 written + 42 skipped = 80 bytes per row.
;   5) At midnight, reverse time: mirror the masks without extra data.
; ---------------------------------------------------------------------------

    mov  al,1
    int  10h                     ; Tiny mode switch (assumes AH=0): 40x25 text => 80 bytes/row.

    push 0B800h
    pop  es                      ; Warp gate to VRAM: ES=B800h (direct framebuffer writes).

    mov  di,104h                 ; Spawn point: hand-tuned center on the text sky.
    mov  si,snow                 ; SI = spellbook pointer: 10x 16-bit row masks.
    mov  dx,19                   ; 19 rows total: forward + mirrored rewind.

NORTHPOLE:                       ; One winter "day" per iteration -> emits one snow row
    lodsw                        ; AX <- next rune (16 bits = 16 columns)
    xchg ax,bx                   ; BX <- rune (byte-saver). AX becomes scratch.

    mov  ah,7                    ; Same attribute for all cells: cheap and clean.
    mov  cl,16                   ; 16 bits to chant.

PIXELRUNE:                       ; 1 bit -> 1 text cell (space or '*')
    mov  al,32                   ; Default: silence (space).
    shr  bx,1                    ; Next bit falls into CF (LSB-first stream).
    jnc  SNOWMIST                ; CF=0 => keep silence.
    mov  al,42                   ; CF=1 => '*' crystal appears.

SNOWMIST:
    stosw                        ; Drop the glyph into VRAM (AL+AH). DI += 2.
    loop PIXELRUNE               ; Repeat 16 times: rune becomes a row.

    ; --- ELF TRICK #1: "data compression by theft" ---
    ; No extra bits stored for the extra width: we steal 3 already-drawn cells.
    lea  bx,[di-28]              ; -28h = -40 bytes => 4 cells before row start (perfect sample spot).
    mov  cl,3                    ; Copy 3 cells to extend the snowflake arms.

ELFCLONE:                        ; Mini blitter: VRAM -> VRAM, makes the silhouette richer
    mov  ax,[es:bx]              ; Read an earlier cell from the same row...
    stosw                        ; ...paste it to the right.
    dec  bx
    dec  bx                      ; BX -= 2 (one cell left). Two DECs are byte-cheap.
    loop ELFCLONE

    ; --- ELF TRICK #2: the sleigh stride equation ---
    ; We wrote 19 cells = 38 bytes. Add 42 => 80 bytes exactly (one 40-col row).
    add  di,42                   ; Perfect row advance with zero multiplication.

    ; --- MIDNIGHT MIRROR: 10 runes become 19 rows ---
    ; Above the waist: read forward. Below: rewind 2 runes each row to mirror for free.
    cmp  dl,11                   ; When the countdown crosses the middle...
    jae  MIDNIGHT                ; ...keep going forward in the spellbook.
    sub  si,4                    ; ...or rewind 2 words to mirror (no extra data!).

MIDNIGHT:
    dec  dx                      ; Next winter day (row).
    jnz  NORTHPOLE               ; Until the snowflake is complete.
    ret

snow:
    ; The spellbook: 10 runes, each rune = 16-bit horizontal mask.
    ; Rewind-mirroring turns this small chant into a full 19-row symmetric snowflake.
    dw 0200h,0A80h,4714h,0C218h,0CA9Ch
    dw 2720h,1240h,4A92h,2724h,0FFFFh

; ---------------------------------------------------------------------------
; EPILOGUE
;
; The portal closes. The elf vanishes back into 0100h.
; No animation, no cleanup: the snowflake remains suspended in VRAM,
; a frozen artifact of bitwise spells and shameless ornament cloning.
;
; You never stored the whole winter—only its essence:
;   - 10 runes for shape,
;   - a rewind for symmetry,
;   - a VRAM steal for width,
;   - and a sleigh stride that lands perfectly every time.
;
; Merry VCCC, and may your bytes be ever smaller.
; ---------------------------------------------------------------------------
