\ -----------------------------------------------------------------------------
\  Toemialuekohtane rumputtaja / domain specific beatmaker
\ -----------------------------------------------------------------------------
\
\ Mostly Forth 2012.
\
\ Environmental dependencies are many. Tested on gforth on WSL2 Ubuntu.
\
\ Goes via the domain of demoscene intro sound synthesis to the realm of Savo.
\
\ Sliced and packaged for the Instanssi 2024 demo party Summamutikka competition.
\
\ Another part of my bigger forthcoming (pun, hehe) production.
\
\ Accompanied by a live presentation in Finnish and/or Savo. Stream recording
\ likely available in archives. Look for "Summamutikka" of Instanssi 2024.
\
\
\ This takes a large chunk of memory from the dictionary, so launch like that:
\
\    gforth --dictionary-size 1G
\
\
\ Author: Paavo Nieminen (aka "qma" aka "The Old Dude")
\ 
\


\ -----------------------------------------------------------------------------
\ Floating point and vector math utility wordings; cherry-picked from a larger
\ math library that I need under the bigger forthcoming production.
\ -----------------------------------------------------------------------------

: />F ( n1 n2 -- ) ( F: -- r ) \ r is the floating point value of n1 / n2.
  SWAP S>F S>F F/
;

VARIABLE randstate 1 randstate !
: rand ( -- n )  randstate @ 16807 * DUP randstate ! ;
: randF ( F: -- r )  rand S>F $8000000000000000 S>F F/ ;
\ Store r to sfloat array at index n, leaving array base address unchanged
: SF>ARR ( sf-addr n -- sf-addr ) ( F: r -- ) SFLOATS OVER + SF! ;
: SF!, ( sf-addr1 -- sf-addr2 ) ( F: r -- )  \ store r as sfloat in f-addr1, f-addr2 is next loc.
  DUP SF! SFLOAT+
;
: S>F, ( sf-addr1 n -- sf-addr2 ) \ Store n as sfloat in addr1, addr2 is location of next sfloat
   S>F SF!,
;

: -1f, -1 S>F, ;
:  0f,  0 S>F, ;
:  1f,  1 S>F, ;

: SFvec3!, ( F: r g b -- ) FROT SF!, FSWAP SF!, SF!, ;

\ This will add values to existing ones. Hmm... FIXME: These must be intermediate thoughts leading up to
\ more vector-oriented stuff..
: SFvec3+, ( F: r g b -- ) FROT DUP SF@ F+ SF!, FSWAP DUP SF@ F+ SF!, DUP SF@ F+ SF!, ;

\ Compute r2 = 1/r1; just let it become +/-Inf, if r1 is 0...
: F^-1 ( -- ) ( F: r1 -- r2 ) 1 S>F FSWAP F/ ;

\ Swap sfloats at array indices u1 and u2
: SFSWAP ( sf-addr u1 u2 -- sf-addr )
  2>R DUP R> SFLOATS + OVER R> SFLOATS + ( sf-addr a1 a2 )
  2DUP SF@ SF@ SF! SF!
;

\ Negate single-prec. float at f-addr, advance to next addr.
: @SFNEG!, ( f-addr1 -- f-addr2 ) DUP SF@ FNEGATE DUP SF! SFLOAT+ ;

\ Invert single-prec. float at f-addr, advance to next addr.
: @SFINV!, ( f-addr1 -- f-addr2 ) DUP SF@ F^-1 DUP SF! SFLOAT+ ;

\ -----------------------------------------------------------------------------
\ Midi note numbers as constants
\ Some of them... for planning... this is work-in-progress at Instanssi 2024...

#0   CONSTANT C-0
#1   CONSTANT C#0
#1   CONSTANT Db0
#2   CONSTANT D-0
#3   CONSTANT D#0
#3   CONSTANT Eb0
#4   CONSTANT E-0
#5   CONSTANT F-0
#6   CONSTANT F#0
#6   CONSTANT Gb0
#7   CONSTANT G-0
#8   CONSTANT G#0
#8   CONSTANT Ab0
#9   CONSTANT A-0
#10  CONSTANT A#0
#10  CONSTANT Bb0
#11  CONSTANT B-0
#12  CONSTANT C-1
\ ... just planning the looks of it as of now..
\ generate the rest if the format stays..
#24  CONSTANT C-2
#26  CONSTANT D-2
#28  CONSTANT E-2
#29  CONSTANT F-2
\ ...
#34  CONSTANT Bb2
#35  CONSTANT B-2
#36  CONSTANT C-3
#38  CONSTANT D-3
#40  CONSTANT E-3
#41  CONSTANT F-3
#48  CONSTANT C-4
#50  CONSTANT D-4
\ ...
#52  CONSTANT E-4
#53  CONSTANT F-4
#55  CONSTANT G-4
#57  CONSTANT A-4
#59  CONSTANT B-4
\ ...
#60  CONSTANT C-5
#62  CONSTANT D-5
#64  CONSTANT E-5
#65  CONSTANT F-5
#67  CONSTANT G-5
#69  CONSTANT A-5
#71  CONSTANT B-5
#72  CONSTANT C-6
#84  CONSTANT C-7
#81  CONSTANT A-6
#93  CONSTANT A-7
#105 CONSTANT A-8





\ -----------------------------------------------------------------------------
\ Audio generation lexicon..
\
\ And a Proof-of-concept of music. Just make a buffer of floats to be replayed.
\
\ These things should be quite agnostic to the system used for playback ultimately.
\
\ This should ultimately become a generic softsynth lexicon, and the content creation
\ should diverge from this. So far it is not yet the case.
\ TODO: Work, work, work, on this..
\
\ This is a snapshot published for Instanssi 2024.


\ ------------------------------------------------------------
\ Utilities and other atoms

#48000 CONSTANT SampleRate

: splf>t ( u -- ) ( F: r1 -- r2 ) SampleRate />F F* ; \ r2 is "wave time" at u:th sample at freq r1

\  Tried creating a look-up table for note frequencies in Forth, hoping that it would
\  pack better than the implementation of F** which is used only by note>freq as of writing this.
\  Unfortunately, it turned out to be exactly the same size; so, I'll opt for precise notes.
\  TODO: One more option could be to implement note>freq in assembly (with or without pow)
\  TODO: Yet one more, SDL2 defines "extern DECLSPEC float SDLCALL SDL_powf(float x, float y);"
\  which should be within the reach of my platform API. Use instead of implementing costly F** ?

: note>freq ( u -- ) ( F: -- f ) \ f will be fundamental frequency of midi note u
  2 S>F #69 - 12 />F F** 440 S>F F*
;

\ Oscillators.. pre-fix these "tilde" like "wave"
: ~sin ( -- ) ( F: r1 -- r2 ) FDUP F+ Pi F* FSIN ; \ r1 is phase or "wave time", r2 is output

\ Envelopes.. prefix these "^" like "rising and falling envelope"..
\ e is 1->0 falling envelope of length u1 at sample u2, returns max(e,0) beyond end
: ^fall ( u1 u2 -- ) ( F: -- e ) 2DUP - ROT />F DROP   0 S>F FMAX ;
: ^fall2 ( u1 u2 -- ) ( F: -- e ) ^fall FDUP F* ;
: ^fall3 ( u1 u2 -- ) ( F: -- e ) ^fall2 FDUP F* ;

  \ Fall from 1+u2 to 1, usable in bass drum like things..:
: ^pitchfall3 ( u1 u2 u3 -- ) ( F: -- e ) ROT ROT ^fall3 S>F F* 1 S>F F+ ;

: stereomix ( addr1 -- addr2 ) ( F: sL sR -- )
  DUP DUP SF@ F+ SF! SFLOAT+
  DUP DUP SF@ F+ SF! SFLOAT+
;





\ ------------------------------------------------------------
\ Very generic building blocks: delay unit
\

\ A constant-length delay unit
: cdelay ( samples "name" -- )
  CREATE
     DUP ,   \ length
     0 ,     \ write index
     SFLOATS ALLOT
;

: cdelay! ( delay-addr -- ) ( F: in -- )
  >R
  R@ 1 CELLS + @
  DUP SFLOATS R@ 2 CELLS + + SF!
  1+ R@ @ MOD R> 1 CELLS + !
;

: cdelay@ ( delay-addr -- ) ( F: -- delayed )
  >R
  R@ 1 CELLS + @ 1+ R@ @ MOD SFLOATS R> 2 CELLS + + SF@
;

\ This uses the delay unit at delay-addr as a feedback delay, or "FBCF" comb filter
: cdelay-fb ( delay-addr -- ) ( F: x g -- y )
  DUP cdelay@ F* F+ FDUP cdelay!
;

\ This uses the delay unit at delay-addr as an all-pass filter with gain g
: cdelay-ap ( delay-addr -- ) ( F: x g -- y )
  FDUP FROT ( g g x ) DUP cdelay@ ( g g x M ) FROT ( g x M g ) FOVER F* ( g x M gM )
  FROT F+ ( g M x+gM ) FDUP cdelay!
  FROT F* F- ( M-g[x+gM] )  \ done?
;



\ ----------------------------------------------

\ Directly implemented from https://ccrma.stanford.edu/~jos/pasp/Schroeder_Reverberators.html
\ Original times are said to be for 25kHz sample rate, so I double them.. should pick actual
\ primes, of course. Could play around with these and add more units, too...
\ Cool stuff.. really cool. But takes up a lot of bytes in the exe.. If I go this way, I must
\ really over-over-over-use this for aesthetic effect. Also, this is the first version which could
\ possibly be optimized a lot..

12000 2 *   cdelay long-delay

1051 ( prime near 347 3 * )  cdelay AP1
227  ( prime near 113 2 * )  cdelay AP2
83   ( prime near 37  2 * )  cdelay AP3

3373 ( prime near 1687 2 * ) cdelay FBCF1
3203 ( prime near 1601 2 * ) cdelay FBCF2
4111 ( prime near 2053 2 * ) cdelay FBCF3
4507 ( prime near 2251 2 * ) cdelay FBCF4


\ ---------------------------------------------------------------

: e-3f ( n -- ) ( F: -- r ) 1000 />F ; \ Syntax sugar for floats: r is n times .001



: TryEffect ( sf-addr u -- )  \ Simulate a mock-up effect for u samples starting from sf-addr
  0 DO
    \ Our input is already stereo now, but let us sum it back to mono for effect (for early testing)
    OVER I 2 * SFLOATS  + DUP SF@ DUP 1 SFLOATS + SF@ F+

    400 e-3f long-delay cdelay-fb

    FDUP

    700 e-3f AP1 cdelay-ap
    700 e-3f AP2 cdelay-ap
    700 e-3f AP3 cdelay-ap
    FDUP       ( x x )        773 e-3f FBCF1 cdelay-fb
    FSWAP FDUP ( x1 x x )     802 e-3f FBCF2 cdelay-fb
    FSWAP FDUP ( x1 x2 x x )  753 e-3f FBCF3 cdelay-fb
    FSWAP      ( x1 x2 x3 x ) 733 e-3f FBCF4 cdelay-fb
    ( x1 x2 x3 x4 )
    FROT F+ ( x1 x3 s1 )
    FROT FROT F+ ( s1 s2 )
    FOVER FOVER F- ( s1 s2 OutD )
    FROT FROT F+ ( OutD OutA )

    36 S>F F/        \ attenuation of effect channels .. 
    FSWAP 36 S>F F/

    ( d w1 w2 )

    DUP FROT FROT ( w2 d w1) FOVER F+ ( w2 d w1+d ) SF! F+ 1 SFLOATS + SF!

  LOOP
  DROP
;




\ Atomize like forth can
: FMIXA ( F: a b r -- c ) \ c is (1-r)a + rb = a + r(b-a) = a + rb -ra
  FROT ( b r a ) 
  FOVER ( b r a r ) 
  1 S>F FSWAP F- ( b r a 1-r ) 
  F* ( b r [1-r]a )
  FROT FROT F* ( [1-r]a b*r ) 
  F+ ( [1-r]a + rb )
;




\ ---------------------------------------------------------------

\ Synth and sequencer kinda.. Evolving, step by step..

VARIABLE rec-addr

: stereo-mix-to-rec ( -- ) ( F: sL sR -- ) \ mix a stereo sample at recording point
  rec-addr @
  DUP DUP SF@ F+ SF! SFLOAT+
  DUP DUP SF@ F+ SF! SFLOAT+
  rec-addr !
;

: mono-dup-mix-to-rec ( -- ) ( F: sM -- ) \ mix a mono sample at recording point
  FDUP
  rec-addr @
  DUP DUP SF@ F+ SF! SFLOAT+
  DUP DUP SF@ F+ SF! SFLOAT+
  rec-addr !
;



\ VARIABLE bpm 120 bpm !
\ FIXME: For syncing, I'm gonna need Ftime-in-beats etc..
\ But for this first mock-up, I fix tempo at 120 BPM so 1.0 seconds == 2 beats.

\ Older ideas: instrument is an xt
\ Current ideas: everything rolls through a per-sample simulator call

\ Needs: Proper handling of BPM. LFOs / CC-automation.. combine with sync tracking while at it?

\ State variables of a state machine here.. 
VARIABLE steptime

: steptime! ( u -- )  48000 64 / * steptime ! ; \ sequencer steps become u ticks long

\ Generic synth
0 CONSTANT ~~note
1 CONSTANT ~~level
2 CONSTANT ~~length
3 CONSTANT ~~noiselev
4 CONSTANT ~~pitchenva

5 CONSTANT ~~fmodf
6 CONSTANT ~~fmoda

7 CONSTANT ~~cursamp

CREATE minisyn 8 CELLS ALLOT

\ Do I need integer parameters, or is everything float in practice:
: ~~ipar@ ( u -- n ) CELLS minisyn + @ ;
: ~~ipar! ( u -- n ) CELLS minisyn + ! ;
: ~~ipar+! ( u -- n ) CELLS minisyn + +! ;
\ Floating point parameters ( assume single float storage is at most one cell.. x86-64 has.. )
: ~~fpar@ ( u -- ) ( F: r -- ) CELLS minisyn + SF@ ;
: ~~fpar! ( u -- ) ( F: -- r ) CELLS minisyn + SF! ;

: note  ( note -- ) ~~note ~~ipar! ;
: note+ ( delta -- ) ~~note ~~ipar+! ;
: cnote>freq ( -- ) ~~note ~~ipar@ note>freq ;
: sI ( -- u ) ~~cursamp ~~ipar@ ;

\ A tentative sample-by-sample rendering synth state machine

: ~~pitch ( F: freq -- freq2 ) ~~length ~~ipar@ sI ~~pitchenva ~~ipar@ ^pitchfall3 F* ;
: ~~noise ( F: audio -- audio2 ) randF ~~noiselev ~~fpar@ FMIXA ;
: ~~vol   ( F: audio -- audio2 ) ~~length ~~ipar@ sI ^fall3 F*   ~~level ~~fpar@ F* ;
: ~~phasemod ( F: phase -- phase2 )
  FDUP   ( F: wt wt )
  ~~fmodf ~~fpar@ F* ( F: wt wt2 )
  ~sin ~~length ~~ipar@ sI ^fall F*   ~~fmoda ~~fpar@ F*   F+
;

\ This can make some different kinds of bleeps and bloops.
: synth-sim ( -- )
  cnote>freq  ~~pitch  sI splf>t  ~~phasemod  ~sin  ~~noise  ~~vol
  mono-dup-mix-to-rec
;

: _  ( -- ) steptime @ 0 DO synth-sim 1 ~~cursamp ~~ipar+! LOOP ; \ Eye-pleasing "pause" for one sequencer step
: O  ( -- ) 0 ~~cursamp ~~ipar! _ ;                                \ Eye-pleasing "Onset" for current note
: x_ ( u -- ) 0 DO _ LOOP ;                                        \ Eye-pleasing "pause" for u sequencer steps






: Track1 ( -- ) \ Remember to set start note before call (ex. D-2 note ). 
     ( 00:00 -----> )           O
     ( 00:01        )  2  note+ O
     ( 00:02        )           O
     ( 00:03        )           O
     ( 01:00 -----> )  12 note+ O
     ( 01:01        ) -12 note+ O
     ( 01:02        )           O
     ( 01:03        )           O
     ( 02:00 -----> ) -2  note+ O
     ( 02:01        )  2  note+ O
     ( 02:02        )           O
     ( 02:03        )  12 note+ O
     ( 03:00 -----> ) -12 note+ O
     ( 03:01        )  12 note+ O
     ( 03:02        ) -12 note+ O
     ( 03:03        )   1 note+ O
     ( 04:00 -----> ) -3  note+ O
     ( 04:01        )  2  note+ O
     ( 04:02        )           O
     ( 04:03        )           O
     ( 05:00 -----> )  12 note+ O
     ( 05:01        ) -12 note+ O
     ( 05:02        )           O
     ( 05:03        )           O
     ( 06:00 -----> ) -2  note+ O
     ( 06:01        )  2  note+ O
     ( 06:02        )           O
     ( 06:03        )  12 note+ O
     ( 07:00 -----> ) -12 note+ O
     ( 07:01        )  12 note+ O
     ( 07:02        ) -12 note+ O
     ( 07:03        )  13 note+ O
     ( ended -----> ) -15 note+   \ enable repetition from same starting note again
;

\ Rhythmical patterns could be laid out like this, so they look almost like "piano rolls" :
\ (Current examples are agnostic of pitch state..)
\
\                  00:00 |_______._______|_______._______|_______._______|_______._______| 08:00
: Track4 ( -- )          _ _ _ _ O _ _ _ _ _ _ _ _ _ _ _ _ _ _ _ _ _ _ _ _ _ _ _ _ _ _ _ ;
: Track2 ( -- )          O _ _ _ O _ _ _ O _ _ _ O _ _ _ O _ _ _ O _ _ _ O _ _ _ O O O _ ;
\                        |       :       |       :       |       :       |       :       |


: Track5 ( -- ) \ Remember to set a starting note before calling this..
     ( 00:00 -----> )           _
     ( 00:01        )           O
     ( 00:02        )           _
     ( 00:03        )  10 note+ O
     ( 01:00 -----> )           _
     ( 01:01        )  -2 note+ O
     ( 01:02        )  -2 note+ O
     ( 01:03        )  -1 note+ O
     ( ...          )           24 x_
     ( 07:03        )
     ( ended -----> )  -5 note+ \ enable repeat from same starting note
;

\ The song would look nicer with instrument settings factored out like this.
\ But couldn't use few-at-a-time state changes without losing clarity?
\ Or maybe clarity is not lost.. it's just typing like a story.. it is a continuum.
\ We know that each instrument must come after another. If you change their order,
\ the plot changes accordingly, and edits must be made. And it is clear. And aesthetic...

: |> ( floatbuf -- floatbuf ) DUP rec-addr ! ; \ "From the top", it looks like a play button |>

: Instr01[BD]
    1 5 />F ~~level     ~~fpar!
    0 S>F   ~~fmoda     ~~fpar!
    1 S>F   ~~fmodf     ~~fpar!
    14000   ~~length    ~~ipar!
    3       ~~pitchenva ~~ipar!
    0 S>F   ~~noiselev  ~~fpar!
    D-3 note
;

: Instr02[Bass]
    1 5 />F  ~~level     ~~fpar!
    7 8 />F  ~~fmoda     ~~fpar!
    14000    ~~length    ~~ipar!
    0        ~~pitchenva ~~ipar!
    0 S>F    ~~noiselev  ~~fpar!
    D-2 note
;

: Instr03[Blip]
    1 4 />F  ~~level     ~~fpar!
    14000    ~~length    ~~ipar!
    1 8 />F  ~~fmoda     ~~fpar!
    0        ~~pitchenva ~~ipar!
    0 S>F    ~~noiselev  ~~fpar!
    B-4 note
;

: Instr04[SD]
    1 16 />F ~~level     ~~fpar!
    7 8 />F  ~~fmoda     ~~fpar!
    14000    ~~length    ~~ipar!
    1 S>F    ~~noiselev  ~~fpar!
    0        ~~pitchenva ~~ipar!
;



\ Hmm... Should call them patterns instead of tracks, to be consistent
\ with pattern-based trackers.. shorthand Pat00 Pat01 etc.

: SongOfProd2 ( floatbuf -- )
    8 steptime!

    Instr01[BD]    |> Track2  Track2  Track2  Track2
    Instr02[Bass]  |> Track1  32 x_   Track1
    Instr03[Blip]  |>  32 x_  Track5
    Instr04[SD]    |> Track4

    DUP 960000 TryEffect
    DROP \ For fast tries without much music.. just the idea in few notes
;

: SongOfProd2.alt ( floatbuf -- )
    8 steptime!

    Instr01[BD]   |>  64 x_           Track2  Track2  Track2  Track2
    Instr02[Bass] |>  Track1  Track1  Track1  Track1  Track1  Track1  Track1  Track1
    Instr03[Blip] |>  96 x_                   Track5  Track5  Track5  Track5  Track5  Track5
    Instr04[SD]   |>  128 x_                          Track4  Track4  Track4  Track4


  \ Very preliminary try:
  DUP 4000000 TryEffect

  DROP
;

\ Benchmark packed sizes: Silence 4391, beeps 4417, short without
\ reverb 4550, longer without reverb 4562, longer with reverb 4772
\ From silence to full effect is less than 500 bytes. Totally OK for
\ the synth part. So - reduce other parts, not audio!

\ Reference tries below, for reference. Beeping and silence/zeroinit..
: SongOfProd2.alt.beeping ( floatbuf -- )
    960000 DUP 0 DO  I 10 />F DUP I ^fall 1 S>F F+ F*  FSIN
      4000 I 96000 MOD ^fall F*	OVER I SFLOATS + SF!
    LOOP DROP ( DUP 960000 TryEffect ) DROP ;
: SongOfProd2.alt.silence ( floatbuf -- ) DROP ;



\ Reserve space for some seconds of 48 kHz stereo (2 ch) frames 
VARIABLE audio_length_frames 100 48000 2 * * audio_length_frames !
CREATE audio audio_length_frames @ SFLOATS ALLOT

: demo ( -- )
  \ Pre-compute music:
  audio SongOfProd2.alt
;

\ The following assumes that character is 8 bits, byte order is that of i386 
: shift8b, ( n1 -- n2 ) DUP $ff AND C, 8 RSHIFT ;
: byte, ( n8 -- ) C, ;
: word, ( n16 -- ) shift8b, shift8b, DROP ;
: dword, ( n32 -- ) shift8b, shift8b, word, ;

\ Just make a header for constant-size audio data
CREATE _wav_header
  'R' c, 'I' c, 'F' c, 'F' c,   \ 4 bytes
  audio_length_frames @ SFLOATS 36 + dword,  \ 4 bytes chunk size 4+n where n is bytes in fmt+wave+?
  'W' c, 'A' c, 'V' c, 'E' c,   \ 4 bytes

  'f' c, 'm' c, 't' c, BL c,    \ 4 bytes
  16 dword,                     \ 4 bytes chunk size
  3  word,                      \ 2 bytes format code (3==ieee float)
  2  word,                      \ 2 bytes number of interleaved channels
  48000 dword,                  \ 4 bytes sampling rate in frames per s
  2 4 48000 * * dword,          \ 4 bytes data rate bytes per s
  8 word,                       \ 2 bytes data block size
  32 word,                      \ 2 bytes bits per sample

  'd' c, 'a' c, 't' c, 'a' c,   \ 4 bytes "data"
  audio_length_frames @ SFLOATS dword,  \ 4 bytes chunk size

\ clear the whole audio buffer
: silence ( -- )
  audio audio_length_frames @ SFLOATS ERASE
;

\ write out our audio buffer to ./output.wav
: write ( -- )
  S" output.wav" W/O CREATE-FILE  ABORT" CREATE-FILE FAILED"
  DUP _wav_header 44 ROT WRITE-FILE ABORT" WRITE-FILE FAILED"

  DUP audio audio_length_frames @ SFLOATS ROT WRITE-FILE ABORT" WRITE-FILE FAILED"
  CLOSE-FILE  ABORT" CLOSE-FILE FAILED"
;

\ Default step time

8 steptime!


\ Then translate my user interface into Savo dialect of finland.. from domain specific to
\ culture specific ..

\ ---------------------------------------------
\ I try to do these by hard on-the-fly in the live performance
\ but just in case I freeze, here's the plan of what to show:

: moekkaile ." Hei, Ilimentymjuhula 2024!" CR ;
: toestoo 0 POSTPONE LITERAL POSTPONE DO ; IMMEDIATE
: joskehtoot POSTPONE LOOP ; IMMEDIATE

: kunnolla 10 toestoo moekkaile joskehtoot ;


\ ----------------------------------------------
\ Viele toemialakohtasempi jumputussanasto
\ elikk sama savoksi totteutettuna:

: pasarilla  Instr01[BD]   ;
: virvelilla Instr04[SD]   ;
: passolla   Instr02[Bass] ;

: purkkiin   write ;
: uuestaan   silence audio ;

\ Example, if I should totally forget what to show at live show:
: raetaA   O _ _ _ O _ _ _ O _ _ _ O _ O _ ;
: raetaB   _ _ _ _ O _ _ _ _ O _ _ O _ _ O ;
: raetaC   _ _ O _ _ _ O _ _ _ O _ _ _ O _ ;

: piisi ( floatbuf -- floatbuf )
    |>  pasarilla 3 toestoo raetaA joskehtoot
    |>  passolla   Track1  32 x_   Track1
    Instr03[Blip]  |>  32 x_  Track5
    |> virvelilla Track4 Track4 Track4
;

audio
CR
.( -------------------------------------------------) CR
.( Pavesforthin syntikkamoduuli ladattu.) CR
.( Kaekki on valamiina perjantai-illan jumputtelluun) CR
.( -------------------------------------------------) CR
