#include "data\\shaders\\common.h"
#include "data\\shaders\\input_formats.h"

Texture2D g_color : register(t0);
SamplerState g_sam_linear : register(s3);

#define FXAA_EDGE_THRESHOLD      ( 1.0 / 8.0 )
#define FXAA_EDGE_THRESHOLD_MIN  ( 1.0 / 32.0 )
#define FXAA_SEARCH_STEPS        8
#define FXAA_SEARCH_THRESHOLD    ( 1.0 / 4.0 )
#define FXAA_SUBPIX_CAP          ( 3.0 / 4.0 )
#define FXAA_SUBPIX_TRIM         ( 1.0 / 8.0 )

#define FXAA_SUBPIX_TRIM_SCALE ( 1.0 / ( 1.0 - FXAA_SUBPIX_TRIM ) )

float FxaaLuma( float3 rgb )
{
    return rgb.y * ( 0.587/0.299 ) + rgb.x;
}

float3 FxaaPixelShader( float2 pos, SamplerState smpl, Texture2D tex, float2 rcpFrame )
{
    // Early exit if local contrast below edge detect limit
    float3 rgbN = tex.SampleLevel( smpl, pos.xy, 0.0, int2( 0, -1 ) ).xyz;
    float3 rgbW = tex.SampleLevel( smpl, pos.xy, 0.0, int2( -1, 0 ) ).xyz;
    float3 rgbM = tex.SampleLevel( smpl, pos.xy, 0.0, int2( 0, 0 ) ).xyz;
    float3 rgbE = tex.SampleLevel( smpl, pos.xy, 0.0, int2( 1, 0 ) ).xyz;
    float3 rgbS = tex.SampleLevel( smpl, pos.xy, 0.0, int2( 0, 1 ) ).xyz;
    float lumaN = FxaaLuma( rgbN );
    float lumaW = FxaaLuma( rgbW );
    float lumaM = FxaaLuma( rgbM );
    float lumaE = FxaaLuma( rgbE );
    float lumaS = FxaaLuma( rgbS );
    float rangeMin = min( lumaM, min( min( lumaN, lumaW ), min( lumaS, lumaE ) ) );
    float rangeMax = max( lumaM, max( max( lumaN, lumaW ), max( lumaS, lumaE ) ) );
    float range = rangeMax - rangeMin;
    float3 rgbL = rgbN + rgbW + rgbM + rgbE + rgbS;

    // Compute lowpass
    float lumaL = ( lumaN + lumaW + lumaE + lumaS ) * 0.25;
    float rangeL = abs( lumaL - lumaM );
    float blendL = 0.0;
    if ( range != 0 )
        blendL = max( 0.0, ( rangeL / range ) - FXAA_SUBPIX_TRIM ) * FXAA_SUBPIX_TRIM_SCALE;
    blendL = min( FXAA_SUBPIX_CAP, blendL );

    // Choose vertical or horizontal search
    float3 rgbNW = tex.SampleLevel( smpl, pos.xy, 0.0, int2( -1, -1 ) ).xyz;
    float3 rgbNE = tex.SampleLevel( smpl, pos.xy, 0.0, int2( 1, -1 ) ).xyz;
    float3 rgbSW = tex.SampleLevel( smpl, pos.xy, 0.0, int2( -1, 1 ) ).xyz;
    float3 rgbSE = tex.SampleLevel( smpl, pos.xy, 0.0, int2( 1, 1 ) ).xyz;
    rgbL += ( rgbNW + rgbNE + rgbSW + rgbSE );
    rgbL *= 1.0 / 9.0;
    float lumaNW = FxaaLuma( rgbNW );
    float lumaNE = FxaaLuma( rgbNE );
    float lumaSW = FxaaLuma( rgbSW );
    float lumaSE = FxaaLuma( rgbSE );
    float edgeVert = abs( ( 0.25 * lumaNW ) + ( -0.5 * lumaN ) + ( 0.25 * lumaNE ) ) +
                     abs( ( 0.50 * lumaW  ) + ( -1.0 * lumaM ) + ( 0.50 * lumaE  ) ) +
                     abs( ( 0.25 * lumaSW ) + ( -0.5 * lumaS ) + ( 0.25 * lumaSE ) );
    float edgeHorz = abs( ( 0.25 * lumaNW ) + ( -0.5 * lumaW ) + ( 0.25 * lumaSW ) ) +
                     abs( ( 0.50 * lumaN  ) + ( -1.0 * lumaM ) + ( 0.50 * lumaS  ) ) +
                     abs( ( 0.25 * lumaNE ) + ( -0.5 * lumaE ) + ( 0.25 * lumaSE ) );
    bool horzSpan = edgeHorz >= edgeVert;
    float lengthSign = horzSpan ? -rcpFrame.y : -rcpFrame.x;
    if( !horzSpan )
    {
        lumaN = lumaW;
        lumaS = lumaE;
    }
    float gradientN = abs( lumaN - lumaM );
    float gradientS = abs( lumaS - lumaM );
    lumaN = ( lumaN + lumaM ) * 0.5;
    lumaS = ( lumaS + lumaM ) * 0.5;

    // Choose side of pixel where gradient is highest
    bool pairN = gradientN >= gradientS;
    if( !pairN )
    {
        lumaN = lumaS;
        gradientN = gradientS;
        lengthSign *= -1.0;
    }
    float2 posN;
    posN.x = pos.x + ( horzSpan ? 0.0 : lengthSign * 0.5 );
    posN.y = pos.y + ( horzSpan ? lengthSign * 0.5 : 0.0 );

    gradientN *= FXAA_SEARCH_THRESHOLD;

    float2 posP = posN;
    float2 offNP = horzSpan ? float2( rcpFrame.x, 0.0 ) : float2( 0.0f, rcpFrame.y );
    float lumaEndN = lumaN;
    float lumaEndP = lumaN;
    bool doneN = false;
    bool doneP = false;
    posN += offNP * float2( -1.0, -1.0 );
    posP += offNP * float2( 1.0,  1.0 );

    for( int i = 0; i < FXAA_SEARCH_STEPS; i++ )
    {
        if( !doneN )
            lumaEndN = FxaaLuma( tex.SampleLevel( smpl, posN.xy, 0 ).xyz );
        if( !doneP )
            lumaEndP = FxaaLuma( tex.SampleLevel( smpl, posP.xy, 0 ).xyz );

        doneN = doneN || ( abs( lumaEndN - lumaN ) >= gradientN );
        doneP = doneP || ( abs( lumaEndP - lumaN ) >= gradientN );
        if( doneN && doneP )
            break;
        if( !doneN )
            posN -= offNP;
        if( !doneP )
            posP += offNP;
    }

    // Handle if center is on positive or negative side
    float dstN = horzSpan ? pos.x - posN.x : pos.y - posN.y;
    float dstP = horzSpan ? posP.x - pos.x : posP.y - pos.y;
    bool directionN = dstN < dstP;
    lumaEndN = directionN ? lumaEndN : lumaEndP;

    // Check if pixel is in section of span which gets no filtering
    if( ( ( lumaM - lumaN ) < 0.0) == ( ( lumaEndN - lumaN ) < 0.0 ) )
        lengthSign = 0.0;

    // Compute sub-pixel offset and filter span
    float spanLength = ( dstP + dstN );
    dstN = directionN ? dstN : dstP;
    float subPixelOffset = ( 0.5 + ( dstN * ( -1.0 / spanLength ) ) ) * lengthSign;
    float2 posF = float2( pos.x + ( horzSpan ? 0.0 : subPixelOffset ), pos.y + ( horzSpan ? subPixelOffset : 0.0 ) );
    float3 rgbF = tex.SampleLevel( smpl, posF, 0 ).xyz;

		return lerp( rgbF, rgbL, blendL );
}

float4 main(VertexTOut pin) : SV_Target
{
  float2 rcpFrame;
	g_color.GetDimensions(rcpFrame.x,rcpFrame.y);
	rcpFrame.x = 1.0/rcpFrame.x;
	rcpFrame.y = 1.0/rcpFrame.y;

  float4 color = g_color.Sample(g_sam_linear, pin.uv);
  color.xyz = FxaaPixelShader(pin.uv, g_sam_linear, g_color, rcpFrame);
  return float4(color.xyz,1.0f);
}
