#!/bin/bash
set -ex

PKG="$(cargo run --bin pkg)"

# Clean up zip file
ZIP="release-v$(cargo run --bin version).zip"
rm -f "${ZIP}"

# Compile shaders
cargo run --bin compile-shaders --features shaderc

# Build binary for all specified hosts
while [[ -n "${1}" ]]; do
  # Check for --features
  if [[ "${1}" == "--features" ]]; then
    FEATURES="${2}"
    shift 2
    continue
  fi

  # Check for --manifest
  if [[ "${1}" == "--manifest" ]]; then
    MANIFEST="${2}"
    shift 2
    continue
  fi
  
  IFS="-" read -r MACHINE VENDOR OS <<< "${1}"
  VENDOR="${VENDOR:-unknown}"
  OS="${OS:-linux-gnu}"

  # Define dpkg architecture name
  case "${MACHINE}" in
    aarch64) DEBIAN_ARCH=arm64 ;;
    x86_64) DEBIAN_ARCH=amd64 ;;
    *) echo "MACHINE must be one of: aarch64 x86_64"; exit 2 ;;
  esac

  # Define C compiler package name and binary name
  case "${OS}" in
    linux*)
      CCPKG="gcc-${MACHINE//_/-}-${OS}"
      CROSSCC="${MACHINE}-${OS}-gcc"
      ;;
    windows-gnu)
      CCPKG="gcc-mingw-w64-${MACHINE//_/-}"
      CROSSCC="${MACHINE}-w64-mingw32-gcc"
      ;;
    *) echo "Unsupported OS '${OS}'"; exit 3; ;;
  esac
  
  podman build \
    --arch "$(uname -m)" \
    --tag "${PKG}-${MACHINE}-${VENDOR}-${OS}" . \
    --build-arg MACHINE="${MACHINE}" \
    --build-arg VENDOR="${VENDOR}" \
    --build-arg OS="${OS}" \
    --build-arg ARCH="${DEBIAN_ARCH}" \
    --build-arg CCPKG="${CCPKG}"
  podman run \
    --arch "$(uname -m)" \
    --rm \
    --userns keep-id \
    --volume "${PWD}":"/usr/src/${PKG}" \
    --workdir "/usr/src/${PKG}" \
    --env RUSTFLAGS="-C linker=${CROSSCC}" \
    "${PKG}-${MACHINE}-${VENDOR}-${OS}" \
    cargo build \
    --target "${MACHINE}-${VENDOR}-${OS}" \
    --release \
    --locked \
    --no-default-features \
    --features "${FEATURES}"
  
  # Package to zip file
  BINARYPATH="target/${MACHINE}-${VENDOR}-${OS}/release"
  BINARY="${PKG}"
  RENAMEDBIN="demo.${MACHINE}"
  if [[ "${OS}" == "windows"* ]]; then
    BINARY="${BINARY}.exe"
    RENAMEDBIN="demo.exe"
  fi
  cp "${BINARYPATH}/${BINARY}" "${BINARYPATH}/${RENAMEDBIN}"
  zip -u -j "${ZIP}" "${BINARYPATH}/${RENAMEDBIN}"

  shift
done

# Package config and resources to zip file
if [[ -n "${MANIFEST}" ]]; then
  zip -u "${ZIP}" -@ < "${MANIFEST}"
else
  zip -u -r "${ZIP}" data/
fi

zip -u "${ZIP}" config.ron

# Package sources
tar -cvzf /tmp/src.tar.gz \
  buildrelease.sh \
  Dockerfile \
  Cargo.toml \
  Cargo.lock \
  src/ \
  shaders/
zip -u -j "${ZIP}" /tmp/src.tar.gz
