#version 460

#define GEOMETRY_INFORMATION_STATIC 1

#include <shaders/materials/commons.glsl>
#include <shaders/commons_hlsl.glsl>
#include <shaders/materials/noise/noise3d.glsl>
#include <shaders/materials/commons_instancing_buffers.h>

uniform sampler2D s_NoiseRGBA;

layout(std140, row_major) uniform TransformParamsBuffer{
	EntityTransformParams transform_params;
};

layout(std430) readonly buffer VTXCoords
{
	float coords[];
} vtx_coords;

vec3 vtx_coords_load(uint idx)
{
	idx *= geometry_information.attributes[GAT_Coord].stride;
	idx += geometry_information.attributes[GAT_Coord].offset;
	return vec3(vtx_coords.coords[idx + 0], vtx_coords.coords[idx + 1], vtx_coords.coords[idx + 2]);
}

layout(std430) readonly buffer VTXNormals
{
	float normals[];
} vtx_normals;

vec3 vtx_normals_load(uint idx)
{
	idx *= geometry_information.attributes[GAT_Normal].stride;
	idx += geometry_information.attributes[GAT_Normal].offset;
	return vec3(vtx_normals.normals[idx + 0], vtx_normals.normals[idx + 1], vtx_normals.normals[idx + 2]);
}

layout(std430) readonly buffer VTXAlbedo
{
	float albedo[];
} vtx_albedo;

vec3 vtx_albedo_load(uint idx)
{
	idx *= geometry_information.attributes[GAT_Albedo].stride;
	idx += geometry_information.attributes[GAT_Albedo].offset;
	return vec3(vtx_albedo.albedo[idx + 0] , vtx_albedo.albedo[idx + 1], vtx_albedo.albedo[idx + 2]);
}

layout(std430) readonly buffer VTXUV0
{
	float uv0[];
} vtx_uv0;

vec2 vtx_uv0_load(uint idx)
{
	idx *= geometry_information.attributes[GAT_UV0].stride;
	idx += geometry_information.attributes[GAT_UV0].offset;
	return vec2(vtx_uv0.uv0[idx + 0], vtx_uv0.uv0[idx + 1]);
}

layout(std430) readonly buffer VTXCustom0
{
	uint custom0[];
} vtx_custom0;

uint vtx_custom0_load(uint idx)
{
	idx *= geometry_information.attributes[GAT_Custom0].stride;
	idx += geometry_information.attributes[GAT_Custom0].offset;
	return vtx_custom0.custom0[idx + 0];
}

#define vtx_builtin_buffers geometry_information.builtin_attribute_mask

struct VertexInput
{
	uint id;
	vec3 pos;
	vec3 norm;
	vec4 color;
	vec2 uv0;
};

layout(location = 1) out struct
{
	vec3 vCoords;
	vec3 vNorm;
	vec3 vWorldNorm;
	vec3 vLocalPos;
	vec3 vCameraRelativeWorldPos;
	vec4 vColor;
	vec2 vUV0;
} vtx_output;

layout(location = 0) out uint instanceID;

// Custom vertex modifiers params

#ifdef INLINE_VERTEX_MODIFIERS_PARAMS_BLOCK
#inline <INLINE_VERTEX_MODIFIERS_PARAMS_BLOCK>
#endif

//

mat3 transpose_mat3(mat3 matrix)
{
    vec3 row0 = matrix[0];
    vec3 row1 = matrix[1];
    vec3 row2 = matrix[2];
    mat3 result = mat3(
        vec3(row0.x, row1.x, row2.x),
        vec3(row0.y, row1.y, row2.y),
        vec3(row0.z, row1.z, row2.z)
    );
    return result;
}

float det(mat2 matrix) {
    return matrix[0].x * matrix[1].y - matrix[0].y * matrix[1].x;
}

mat3 inverse_mat3(mat3 matrix)
{
    vec3 row0 = matrix[0];
    vec3 row1 = matrix[1];
    vec3 row2 = matrix[2];

    vec3 minors0 = vec3(
        det(mat2(row1.y, row1.z, row2.y, row2.z)),
        det(mat2(row1.z, row1.x, row2.z, row2.x)),
        det(mat2(row1.x, row1.y, row2.x, row2.y))
    );
    vec3 minors1 = vec3(
        det(mat2(row2.y, row2.z, row0.y, row0.z)),
        det(mat2(row2.z, row2.x, row0.z, row0.x)),
        det(mat2(row2.x, row2.y, row0.x, row0.y))
    );
    vec3 minors2 = vec3(
        det(mat2(row0.y, row0.z, row1.y, row1.z)),
        det(mat2(row0.z, row0.x, row1.z, row1.x)),
        det(mat2(row0.x, row0.y, row1.x, row1.y))
    );

    mat3 adj = transpose_mat3(mat3(minors0, minors1, minors2));

    return (1.0 / dot(row0, minors0)) * adj;
}

void main() {
	
	uint load_vertex_idx = SYS_VertexIndex;

	if (instance_params.stride > 0)
	{
		uint flipbook_card = instance_flipbook_card[SYS_InstanceIndex];

		// NOTE: Do we want to wrap around? This seems like needles ALU cost
		flipbook_card = min(flipbook_card, geometry_information.flipbook_cards_num - 1);
		load_vertex_idx += geometry_information.vtx_num * flipbook_card;
	}

	VertexInput vtx_input;
	vtx_input.id    = SYS_VertexIndex;
	vtx_input.pos   = vtx_coords_load(load_vertex_idx);
	vtx_input.norm  = vec3(1.0, 0.0, 0.0);
	vtx_input.color = vec4(1.0);
	vtx_input.uv0   = vtx_input.pos.xy;

	if ((vtx_builtin_buffers & (1 << GAT_Albedo)) != 0)
		vtx_input.color.rgb = vtx_albedo_load(SYS_VertexIndex);

	if ((vtx_builtin_buffers & (1 << GAT_Normal)) != 0)
		vtx_input.norm = vtx_normals_load(load_vertex_idx);

	if ((vtx_builtin_buffers & (1 << GAT_UV0)) != 0)
		vtx_input.uv0 = vtx_uv0_load(SYS_VertexIndex);

#if 1
#ifdef INLINE_VERTEX_MODIFIERS_TRANSFORM_LOCAL_BLOCK
	{
		mat4 mat_instance_model;
		mat_instance_model[0] = vec4(1.0, 0.0, 0.0, 0.0);
		mat_instance_model[1] = vec4(0.0, 1.0, 0.0, 0.0);
		mat_instance_model[2] = vec4(0.0, 0.0, 1.0, 0.0);
		mat_instance_model[3] = vec4(0.0, 0.0, 0.0, 1.0);

		mat4 mat_instance_model_inv = mat_instance_model;

		if (instance_params.stride > 0)
		{
			vec4 inst_m0 = instance_transform[SYS_InstanceIndex * instance_params.stride + 0];
			vec4 inst_m1 = instance_transform[SYS_InstanceIndex * instance_params.stride + 1];
			vec4 inst_m2 = instance_transform[SYS_InstanceIndex * instance_params.stride + 2];

			mat_instance_model[0].xyz = vec3(inst_m0.x, inst_m1.x, inst_m2.x);
			mat_instance_model[1].xyz = vec3(inst_m0.y, inst_m1.y, inst_m2.y);
			mat_instance_model[2].xyz = vec3(inst_m0.z, inst_m1.z, inst_m2.z);
			mat_instance_model[3].xyz = vec3(inst_m0.w, inst_m1.w, inst_m2.w);

			mat_instance_model_inv = mat_instance_model;
			{
				mat3 inv = inverse_mat3(mat3(mat_instance_model_inv));
				//mat3 inv = mat3(transform_params.mModelInv);
				mat_instance_model_inv[0].xyz = inv[0].xyz;
				mat_instance_model_inv[1].xyz = inv[1].xyz;
				mat_instance_model_inv[2].xyz = inv[2].xyz;
				//mat_instance_model_inv = transform_params.mModelInv;
				mat_instance_model_inv[3].x = -(inv[0].x * mat_instance_model[3].x + inv[1].x * mat_instance_model[3].y + inv[2].x * mat_instance_model[3].z);
				mat_instance_model_inv[3].y = -(inv[0].y * mat_instance_model[3].x + inv[1].y * mat_instance_model[3].y + inv[2].y * mat_instance_model[3].z);
				mat_instance_model_inv[3].z = -(inv[0].z * mat_instance_model[3].x + inv[1].z * mat_instance_model[3].y + inv[2].z * mat_instance_model[3].z);
			}

		}

		VertexInput vtx_modifier_input_base = vtx_input; // if we need a copy for anything

		ModifierFactor modifier_factor = modifier_factor_defaults();
		modifier_factor.factor      = 1.0;
		modifier_factor.hash        = uint(SYS_VertexIndex);
		modifier_factor.id          = uint(SYS_VertexIndex);
		modifier_factor.instance_id = uint(SYS_InstanceIndex);
		if (instance_params.stride > 0)
		{
			modifier_factor.instance_id = instance_hash[SYS_InstanceIndex];
		}
		modifier_factor.position    = vtx_input.pos;
		modifier_factor.normal      = vtx_input.norm;
		modifier_factor.color       = vtx_input.color;
		if ((vtx_builtin_buffers & (1 << GAT_Custom0)) != 0)
			modifier_factor.custom0 = vtx_custom0_load(SYS_VertexIndex);
		if ((vtx_builtin_buffers & (1 << GAT_UV0)) != 0)
			modifier_factor.uv0 = vtx_uv0_load(SYS_VertexIndex);
	
		modifier_factor.is_spawned  = false;

		CoordinateSystemTrasforms cs_transforms;
		cs_transforms.mat_local_to_model     = transform_params.mModel;
		cs_transforms.mat_local_to_instance  = mat_instance_model;
		cs_transforms.mat_local_to_model_inv = transform_params.mModelInv;

#inline <INLINE_VERTEX_MODIFIERS_TRANSFORM_LOCAL_BLOCK>

		//vtx_input.color.rgb = vec3(modifier_factor.factor, 0.0, 0.0);

	}
#endif
#endif

	// NOTE: Currently we push pre-instancing as local position
	vtx_output.vLocalPos = vtx_input.pos;

	instanceID = SYS_InstanceIndex;

	// NOTE: Instance transformation block is located here so we
//#define APPLY_INSTANCING_AFTER_MODEL 1
#ifndef APPLY_INSTANCING_AFTER_MODEL
	if (instance_params.stride > 0)
	{
		vec4 inst_m0 = instance_transform[SYS_InstanceIndex * instance_params.stride + 0];
		vec4 inst_m1 = instance_transform[SYS_InstanceIndex * instance_params.stride + 1];
		vec4 inst_m2 = instance_transform[SYS_InstanceIndex * instance_params.stride + 2];

		vec4 v = vec4(vtx_input.pos, 1.0);
		vtx_input.pos.x = dot(inst_m0, v);
		vtx_input.pos.y = dot(inst_m1, v);
		vtx_input.pos.z = dot(inst_m2, v);

		// this is building transposed matrix (which is NOT correct, but maybe enough)
		vtx_input.norm = normalize(vector_transform_by_mat33(vtx_input.norm, transpose(mat4(inst_m0, inst_m1, inst_m2, vec4(0.0, 0.0, 0.0, 1.0)))));

		vec4 inst_c;
		inst_c.rg = unpackHalf2x16(instance_color[SYS_InstanceIndex * 2 + 0]);
		inst_c.ba = unpackHalf2x16(instance_color[SYS_InstanceIndex * 2 + 1]);
		vtx_input.color.rgb = vtx_input.color.rgb * inst_c.rgb;
		//vtx_input.color.rgb = vec3(1.0, 1.0, 1.0);
		vtx_input.color.a   = vtx_input.color.a   * inst_c.a;
	}

	// NOTE: Currently we push pre-instancing as local position
	// vtx_output.vLocalPos = vtx_input.pos;

	vec3 vPosModel = vector_transform_by_mat43(vtx_input.pos, transform_params.mModel);
	vtx_output.vWorldNorm = vector_transform_by_mat33(vtx_input.norm, transform_params.mModelNormal);
#else
	// NOTE: Currently we push pre-instancing as local position
	// vtx_output.vLocalPos = vtx_input.pos;

	{
		mat4 model_position_only = transform_params.mModel;
		model_position_only[0].xyz = vec3(1.0, 0.0, 0.0);
		model_position_only[1].xyz = vec3(0.0, 1.0, 0.0);
		model_position_only[2].xyz = vec3(0.0, 0.0, 1.0);
		
		vtx_input.pos = vector_transform_by_mat43(vtx_input.pos, model_position_only);
		
		if (instance_params.stride > 0)
		{
			vec4 inst_m0 = instance_transform[SYS_InstanceIndex * instance_params.stride + 0];
			vec4 inst_m1 = instance_transform[SYS_InstanceIndex * instance_params.stride + 1];
			vec4 inst_m2 = instance_transform[SYS_InstanceIndex * instance_params.stride + 2];

			vec4 v = vec4(vtx_input.pos, 1.0);
			vtx_input.pos.x = dot(inst_m0, v);
			vtx_input.pos.y = dot(inst_m1, v);
			vtx_input.pos.z = dot(inst_m2, v);

		// this is building transposed matrix (which is NOT correct, but maybe enough)
			vtx_input.norm = normalize(vector_transform_by_mat33(vtx_input.norm, transpose(mat4(inst_m0, inst_m1, inst_m2, vec4(0.0, 0.0, 0.0, 1.0)))));

			vec4 inst_c;
			inst_c.rg = unpackHalf2x16(instance_color[SYS_InstanceIndex * 2 + 0]);
			inst_c.ba = unpackHalf2x16(instance_color[SYS_InstanceIndex * 2 + 1]);
			vtx_input.color.rgb = vtx_input.color.rgb * inst_c.rgb;
			vtx_input.color.a = vtx_input.color.a * inst_c.a;
		}
	}

	// only rotation/scale, position is applied
	mat4 model_rotation_scale_only = transform_params.mModel;
	model_rotation_scale_only[0].w = 0.0;
	model_rotation_scale_only[1].w = 0.0;
	model_rotation_scale_only[2].w = 0.0;
	
	vec3 vPosModel = vector_transform_by_mat43(vtx_input.pos, model_rotation_scale_only);
	vtx_output.vWorldNorm = vector_transform_by_mat33(vtx_input.norm, transform_params.mModelNormal);

	
#endif
	
	// TODO: Make fast path when we can combine model and view. Now we inject us between them because we need
	// model transform for modifiers
#if 0
#ifdef INLINE_VERTEX_MODIFIERS_TRANSFORM_LOCAL_BLOCK
	{
		vtx_input.pos = vPosModel;
		vtx_input.norm = normalize(vtx_output.vWorldNorm);

#inline <INLINE_VERTEX_MODIFIERS_TRANSFORM_LOCAL_BLOCK>
		
		vPosModel = vtx_input.pos;
		vtx_output.vWorldNorm = vtx_input.norm;
	}
#endif
#endif

	vec3 vPosModelView = vector_transform_by_mat43(vPosModel.xyz, transform_params.mView);

	vtx_output.vCoords = vPosModelView;
	vtx_output.vCameraRelativeWorldPos = vPosModel - transform_params.vCameraPosition;
	
	vtx_output.vNorm.x = dot(transform_params.mModelViewInvTrans[0].xyz, vtx_input.norm);
	vtx_output.vNorm.y = dot(transform_params.mModelViewInvTrans[1].xyz, vtx_input.norm);
	vtx_output.vNorm.z = dot(transform_params.mModelViewInvTrans[2].xyz, vtx_input.norm);
	vtx_output.vNorm   = vtx_input.norm;
	vtx_output.vColor  = vtx_input.color;
	vtx_output.vUV0    = vtx_input.uv0;

#ifdef MATERIAL_INFINITE_PLANE

	if (SYS_VertexIndex < 4)
	{
#if 0
#if 0
		if (SYS_VertexIndex == 0)
			vtx_output.vCoords.xyz = vec3(1.0, 0.0, 0.0);
		if (SYS_VertexIndex == 1)
			vtx_output.vCoords.xyz = vec3(0.0, 1.0, 0.0);
		if (SYS_VertexIndex == 2)
			vtx_output.vCoords.xyz = vec3(-1.0, 0.0, 0.0);
		if (SYS_VertexIndex == 3)
			vtx_output.vCoords.xyz = vec3(0.0, -1.0, 0.0);
#else
		if (SYS_VertexIndex == 0)
			vtx_output.vCoords.xyz = vec3(-1.0, 0.0, -1.0);
		if (SYS_VertexIndex == 1)
			vtx_output.vCoords.xyz = vec3(1.0, 0.0, -1.0);
		if (SYS_VertexIndex == 2)
			vtx_output.vCoords.xyz = vec3(1.0, 0.0, 1.0);
		if (SYS_VertexIndex == 3)
			vtx_output.vCoords.xyz = vec3(-1.0, 0.0, 1.0);
#endif
#else
		vPosModel = vector_transform_by_mat33(vtx_input.pos, transform_params.mModel);
		vtx_output.vCoords.xyz = vector_transform_by_mat33(vPosModel.xyz, transform_params.mView);
#endif
		gl_Position = transform_params.mProjection * vec4(vtx_output.vCoords, 0.0);
	}
	else
	{
		vtx_output.vCoords.xyz = vector_transform_by_mat43(vPosModel.xyz, transform_params.mView);
		gl_Position = transform_params.mProjection * vec4(vtx_output.vCoords, 1.0);
	}

#else
	gl_Position = vector_transform_by_mat_projection(vtx_output.vCoords, transform_params.mProjection);
#endif

}