// Adapt physics systems tm
#pragma once

class PhysicsParticleCollection;

struct ForceWithPos {
   D3DXVECTOR3 m_force;
   D3DXVECTOR3 m_pos;
};

class PhysicsParticle {
private:
  D3DXVECTOR3 m_position;
  D3DXVECTOR3 m_prevPosition;
  D3DXVECTOR3 m_positionCandidate;
  D3DXVECTOR3 m_normal;
  D3DXVECTOR3 m_normalRot;
  D3DXVECTOR3 m_positionBase;
  D3DXVECTOR3 m_positionInit;
  D3DXVECTOR3 m_speed;
  D3DXVECTOR3 m_prevSpeed;
  D3DXVECTOR3 m_force;
  D3DXVECTOR3 m_collisionForce;
  float m_mass;
  float m_diameter;

  PhysicsParticleCollection *m_pBelongsToCollection;
public:

 // std::vector<ForceWithPos> m_collisionForcesWithPos;

  PhysicsParticle(float m, const D3DXVECTOR3 *p, float diameter, PhysicsParticleCollection *pBelongsToCollection) {
    m_mass = m;
    m_position = *p;
    m_prevPosition = *p;
    m_positionBase = *p;
    m_positionInit = *p;
    m_diameter = diameter;
    m_speed = D3DXVECTOR3(0.0f, 0.0f, 0.0f);
    m_prevSpeed = D3DXVECTOR3(0.0f, 0.0f, 0.0f);
    m_force = D3DXVECTOR3(0.0f, 0.0f, 0.0f);
    m_collisionForce = D3DXVECTOR3(0.0f, 0.0f, 0.0f);
    m_pBelongsToCollection = pBelongsToCollection;
    m_normal = D3DXVECTOR3(0.0f, 0.0f, 0.0f);
    m_normalRot = D3DXVECTOR3(0.0f, 0.0f, 0.0f);
  }
  void SetPosition(const D3DXVECTOR3 *p) {
    m_prevPosition = m_position;
    m_position = *p;
  }
  void SetNormal(const D3DXVECTOR3 *p) {
    m_normal = *p;
    m_normalRot = *p;
  }
  void SetNormalRot(const D3DXVECTOR3 *p) {
    m_normalRot = *p;
  }
  void SetPositionCandidate(const D3DXVECTOR3 *p) {
    m_positionCandidate = *p;
  }
  void SetSpeed(const D3DXVECTOR3 *p) {
    m_prevSpeed = m_speed;
    m_speed = *p;
  }
  void SetPositionBase(const D3DXVECTOR3 *p) {
    m_positionBase = *p;
  }
  void AddToPositionBase(const D3DXVECTOR3 *p) {
    m_positionBase += *p;
  }
  void RotateBase(float angle, D3DXVECTOR3 *center) {
    D3DXMATRIXA16 matRot;
    D3DXMatrixRotationZ(&matRot, angle);

    D3DXVECTOR4 t = D3DXVECTOR4(m_positionBase.x-center->x, m_positionBase.y-center->y, m_positionBase.z-center->z, 0.0f);
    D3DXVECTOR4 p;
    D3DXVec4Transform(&p, &t, &matRot);
    
    m_positionBase.x = p.x+center->x;
    m_positionBase.y = p.y+center->y;
    m_positionBase.z = p.z+center->z;
  }

  void ClearForce() { 
    m_force.x = 0.0f;
    m_force.y = 0.0f;
    m_force.z = 0.0f;
    m_collisionForce.x = 0.0f;
    m_collisionForce.y = 0.0f;
    m_collisionForce.z = 0.0f;
 //   m_collisionForcesWithPos.clear();
  }
  void AddForce(const D3DXVECTOR3 *f) {
    m_force += *f;
  }
  void AddCollisionForce(const D3DXVECTOR3 *f) {
    m_collisionForce += *f;
  }
  void AddCollisionForceWithPos(const D3DXVECTOR3 *f, const D3DXVECTOR3 *p) {
    ForceWithPos fwp;
    fwp.m_force = *f;
    fwp.m_pos = *p;
//    m_collisionForcesWithPos.push_back(fwp);
  }
  const D3DXVECTOR3* GetPosition() { return &m_position; }
  const D3DXVECTOR3* GetPrevPosition() { return &m_prevPosition; }
  const D3DXVECTOR3* GetNormal() { return &m_normal; }
  const D3DXVECTOR3* GetNormalRot() { return &m_normalRot; }
  const D3DXVECTOR3* GetPositionCandidate() { return &m_positionCandidate; }



  const D3DXVECTOR3* GetPositionBase() { return &m_positionBase; }
  const D3DXVECTOR3* GetSpeed() { return &m_speed; }
  const D3DXVECTOR3* GetPrevSpeed() { return &m_prevSpeed; }
  const D3DXVECTOR3* GetForce() { return &m_force; }
  const D3DXVECTOR3* GetCollisionForce() { return &m_collisionForce; }
  const D3DXVECTOR3* GetPositionInit() { return &m_positionInit; }


  float GetMass() { return m_mass; }
  float GetDiameter() { return m_diameter; }
  float GetDiameterSq() { return m_diameter*m_diameter; }

  PhysicsParticleCollection *GetPtrBelongsToCollection() { return m_pBelongsToCollection; }
};

class PhysicsGridCell {
private:
  std::vector<PhysicsParticle *> m_physicsParticles;

  int m_minContactLevel;
public:
  PhysicsGridCell() {
 //   m_physicsParticles.reserve(4);
    m_minContactLevel = 0;
  }
  ~PhysicsGridCell() {}

  void Clear() {
    if (m_physicsParticles.size() > 10) {
      char tempStr[256];
      sprintf(tempStr, "Physics particles in cell %d\n", m_physicsParticles.size());
      OutputDebugString(tempStr);
    }
    m_physicsParticles.clear();
  }
  void AddParticle(PhysicsParticle *pp) { m_physicsParticles.push_back(pp); }
  int GetMinContactLevel() { return m_minContactLevel; }
  void CalculateMinContactLevel();
  std::vector<PhysicsParticle *> *GetPhysicsParticles() { return &m_physicsParticles; }
  
};

struct PhysicsForce {
  D3DXVECTOR3 m_acceleration;
};

// Physics Grid is n dimensional grid holding certain amount of cells
// The cells itself hold PhysicsGridCell entities, which contain PhysicsParticle pointers
class PhysicsGrid {
private:
//  std::vector< std::vector<PhysicsGridCell> >  m_grid;
  PhysicsGridCell m_grid[1024][1024];
  int m_cellsX, m_cellsY;
  float m_cellWidth, m_cellHeight;

  std::vector< PhysicsForce > m_forces;

  float m_timeStep;

public:

  static unsigned __stdcall Thread1(void *pThis) {
    return ((PhysicsGrid*)pThis)->Thread1();
  }
  unsigned Thread1();

  static unsigned __stdcall Thread2(void *pThis) {
    return ((PhysicsGrid*)pThis)->Thread2();
  }
  unsigned Thread2();

  static unsigned __stdcall Thread3(void *pThis) {
    return ((PhysicsGrid*)pThis)->Thread3();
  }
  unsigned Thread3();


  static unsigned __stdcall Thread4(void *pThis) {
    return ((PhysicsGrid*)pThis)->Thread4();
  }
  unsigned Thread4();

  struct CalculateDataPF {
    std::vector< PhysicsForce > *pForces;
    PhysicsGrid *pPhysicsGrid;
    PhysicsParticleCollection *pPPC;
    std::map<PhysicsParticleCollection*, bool > *pColliding;
    int iteration;
    int numIterations;
    int level;
    int numLevels;
    float timeStep;
    int updateMode;
  };
  std::list<CalculateDataPF> m_dataQueue1;
  std::list<CalculateDataPF> m_dataQueue2;
  std::list<CalculateDataPF> m_dataQueue3;
  std::list<CalculateDataPF> m_dataQueue4;


  void InitGrid(int cellsX, int cellsY, float cellWidth, float cellHeight, float timeStep) {
  //  m_grid.clear();
/*
    for (int y=0; y<cellsY; y++) {
      std::vector<PhysicsGridCell> row;
      for (int x=0; x<cellsX; x++) {
        PhysicsGridCell pgc;
        row.push_back(pgc);
      }
      m_grid.push_back(row);
    }
*/
    m_cellsX = cellsX;
    m_cellsY = cellsY;
    m_cellWidth = cellWidth;
    m_cellHeight = cellHeight;

    m_timeStep = timeStep;
  }

  void ClearForces() {
    m_forces.clear();
  }

  void AddStaticDirectionalForce(D3DXVECTOR3 acceleration) {
    PhysicsForce pf;
    pf.m_acceleration = acceleration;
    m_forces.push_back(pf);
  }

  // returns the current cell pointer matching the given coordinate
  PhysicsGridCell* GetCellPtrAtPos(const D3DXVECTOR3 *p, int *retCellX=NULL, int *retCellY=NULL) {
    int cellIndexUnscrambledY = ((int)(((int)((p->y)/m_cellHeight))))&(m_cellsY-1);
    int cellIndexUnscrambledX = ((int)(((int)((p->x)/m_cellWidth))))&(m_cellsX-1);

    int cellIndexY = ((int)(((int)((p->y)/m_cellHeight))+(m_cellsY<<8)))&(m_cellsY-1);
    int cellIndexX = ((int)(((int)((p->x)/m_cellWidth))+(m_cellsX<<8)+cellIndexY*1*(111321)))&(m_cellsX-1);
    if (retCellX) {
      *retCellX = cellIndexUnscrambledX;
    }
    if (retCellY) {
      *retCellY = cellIndexUnscrambledY;
    }
    return &m_grid[cellIndexY][cellIndexX];
  }

  PhysicsGridCell* GetCellPtrAtCell(int cellIndexX, int cellIndexY) {
    cellIndexY = ((int)(cellIndexY+(m_cellsY<<8)))&(m_cellsY-1);
    cellIndexX = ((int)(cellIndexX+(m_cellsX<<8)+cellIndexY*1*(111321)))&(m_cellsX-1);
    return &m_grid[cellIndexY][cellIndexX];
  }


  void Clear() {
    for (int y=0; y<m_cellsY; y++) {
      for (int x=0; x<m_cellsX; x++) {
        m_grid[y][x].Clear();
      }
    }
  }

  void AddParticle(PhysicsParticle *ppPtr) {
    PhysicsGridCell *pgcPtr = GetCellPtrAtPos(ppPtr->GetPosition());
    pgcPtr->AddParticle(ppPtr);
  }

  void ClearParticleForces(std::vector<PhysicsParticleCollection*> *pParticleCollections);

  int CalculateContactDistances(std::vector<PhysicsParticleCollection*> *pParticleCollections); // returns the biggest encountered contact level

  void CalculateMinContactLevels();

  // for each grid cell, check out the distance for particles in the cell and around the cell
  // if near enough, apply collision force for the particle collection which the particle belongs to
  void CalculateParticleForces(std::vector<PhysicsParticleCollection*> *pParticleCollections, std::map<PhysicsParticleCollection*, bool > *pColliding, int iteration, int numIterations, int level, int numLevels, float timeStep, int updateMode);

  float GetTimeStep() { return m_timeStep; }
};

class PhysicsParticleCollection {
private:

  D3DXVECTOR3 m_currentCenter;
  D3DXVECTOR3 m_currentCenterBase;
  D3DXVECTOR3 m_centerInit;


  D3DXVECTOR3 m_position;
  D3DXVECTOR3 m_speed;
  D3DXVECTOR3 m_force;
  float m_totalMass;

  float m_angle;
  float m_angleSpeed;

  D3DXVECTOR3 m_scale;

  float m_timeOfInit;
  
  //     dynamic: particles interact fully with the environment
  // non dynamic: particles have a static position, but interact can apply forces to other collections which are dynamic
  bool m_bDynamic;

  bool m_bResting;
  int m_restingForFrames;

  D3DXMATRIXA16 m_inertiaTensor0;
  bool m_bInertiaTensorInited;


  // calculate the movement of the mass center from the base
  D3DXVECTOR3 CalculateTransformFromBase();
  // calculate the rotated angle compared to the base
  float CalculateRotationAngleFromBase();
  // calculate the rotated angle compared to the initial position
  float CalculateRotationAngleFromInit();


public:
  int m_collisionForceCount;
  std::vector<PhysicsParticle> m_physicsParticles;

  PhysicsParticleCollection *m_primaryContactWith;
  int m_contactDistanceNumber;
  std::map<PhysicsParticleCollection*, bool> m_mapInContactWith;

  // calculate the mass center point for all of the particles in this collection
  // (current value is saved to m_currentCenter)
  D3DXVECTOR3 CalculateCenter();

  void UpdateParticleSpeedPos(bool bUpdatePos, float timeStep); // update the particle speed & positions based on the body position & orientation

  void InitInertiaTensor();

  PhysicsParticleCollection() {
    m_restingForFrames = 0;
    m_bInertiaTensorInited = false;
  }
  ~PhysicsParticleCollection() {}


  float GetTimeFromInit();
  void InitFromTexture(Texture *t, D3DXVECTOR3 *initialPosition, float scaleX, float scaleY, float scaleZ, int particlesX, int particlesY, bool bDynamic, float totalMass, float initialAngle);

  // Add all particle pointers from this collection to the given grid
  // (forces for particles will be calculated by the grid, also clearing of the forces is handled by the grid)
  void AddParticlesToGrid(PhysicsGrid *pgPtr);

  void CalculateTotalMass();
  // Apply forces to all particles in this collection
  void CalculateParticleForces(std::vector< PhysicsForce > *pForces, PhysicsGrid *pPhysicsGrid, std::map<PhysicsParticleCollection*, bool > *pColliding, int iteration, int numIterations, int level, int numLevels, float timeStep, int updateMode);
  void ApplyForces(float timeStep, float maxMoveDelta, int updateMode, int iteration, int numIterations, int level, int numLevels);

  void AddUniformForce(const D3DXVECTOR3 *force);
  void AddUniformImpulse(const D3DXVECTOR3 *force, float timeStep);

  D3DXVECTOR3 *GetSpeed() { return &m_speed; }
  void SetSpeed(const D3DXVECTOR3 *s) { m_speed = *s; }
  void SetAngularSpeed(float f) { m_angleSpeed = f; }
  void SetIsDynamic() { m_bDynamic = true; }

  const D3DXVECTOR3 *GetCurrentCenter() { return &m_currentCenter; }
  const D3DXVECTOR3 *GetPosition() { return &m_position; }
  float GetCurrentAngle() { return m_angle; }

  // this is for the representative surface mesh only
  const D3DXVECTOR3 *GetScale() { return &m_scale; }

  void SetPosition(const D3DXVECTOR3 *p) {
    m_position = *p;
  }
  void SetAngle(float a) {
    m_angle = a;
  }

  void MarkAsNotResting() {
    m_bResting = false;
  }
  void MarkAsResting() {
    m_bResting = true;
  }
  bool GetIsResting() {
    return m_bResting;
  }

  bool GetIsDynamic() {
    return m_bDynamic;
  }

  float GetTotalMass() {
    return m_totalMass;
  }

};


// class Contact

// PhysicsParticleCollection

// Contact graph is needed for being able to easily go through each level (contact distance to fixed bodies)
// in a way of each contact level at a time.

// Generation of contact graph:
//  -For each body A touching some other body B, make a map entry describing contact
//  



float clamp(float a);
void RotateByAngle(float angle, D3DXVECTOR3 *p);
