// EffectLayout.cpp
#define _WIN32_WINNT 0x400
#include <windows.h>
#include <d3d9.h>
#include <d3dx9.h>

#include <string>
#include <sstream>
#include <vector>
#include <list>
#include <map>

#include "math.h"

using std::string;
using std::list;
using std::vector;

#include "tinyxml/tinyxml.h"

#include "common_globals.h"

#include "d3dapp.h"
#include "mesh.h"

#include "EffectLayout.h"
#include "Effect.h"

#include "..\ade\src\timeline.h"

#include "curve.h"

#include "ContextData.h"
#include "..\ade\src\DemoContextData.h"

#include "sync/sync.h"


const EffectParam* EffectParam::getById(std::string id) const {
  for (std::list<EffectParam>::const_iterator it=params.begin(); it!=params.end(); it++) {
    if (strcmp((*it).id.c_str(), id.c_str()) == 0) {
        return &(*it);
    }
    const EffectParam *epChild = (*it).getById(id);
    if (epChild)
      return epChild;
  }
  return NULL;
}

const EffectParam* EffectParam::getByNameAndId(std::string name, std::string id) const {
  for (std::list<EffectParam>::const_iterator it=params.begin(); it!=params.end(); it++) {
    if ((strcmp((*it).id.c_str(), id.c_str()) == 0) &&
        (strcmp((*it).name.c_str(), name.c_str()) == 0)) {
        return &(*it);
    }
    const EffectParam *epChild = (*it).getByNameAndId(name, id);
    if (epChild)
      return epChild;
  }
  return NULL;
}


const EffectParam* EffectParam::get(std::string name, int index) const {
  int num=0;
  for (std::list<EffectParam>::const_iterator it=params.begin(); it!=params.end(); it++) {
    if (strcmp((*it).name.c_str(), name.c_str()) == 0) {
      if (num == index)
        return &(*it);
      num++;
    }
  }

  if (ref != "") {
    const EffectParam *pRef = m_belongsToLayout->getRoot()->getById(ref);

    num=0;
    for (std::list<EffectParam>::const_iterator it=pRef->params.begin(); it!=pRef->params.end(); it++) {
      if (strcmp((*it).name.c_str(), name.c_str()) == 0) {
        if (num == index)
          return &(*it);
        num++;
      }
    }
  }
  

  char errMsg[1024];
  sprintf(errMsg, "Error:\nMissing \"%s\" from effect xml \"%s\"", name.c_str(), m_belongsToLayout->layoutFile.c_str());
  MessageBox(NULL, errMsg, "Error in effect xml", MB_ICONERROR);
  exit(-1);
  return NULL;
}

const int EffectParam::getN(std::string name) const {
  int num = 0;
  for (std::list<EffectParam>::const_iterator it=params.begin(); it!=params.end(); it++) {
    if (strcmp((*it).name.c_str(), name.c_str()) == 0) {
      num++;
    }
  }
  if (num==0) {
    if (ref != "") {
      const EffectParam *pRef = m_belongsToLayout->getRoot()->getById(ref);

      num=0;
      for (std::list<EffectParam>::const_iterator it=pRef->params.begin(); it!=pRef->params.end(); it++) {
        if (strcmp((*it).name.c_str(), name.c_str()) == 0) {
          num++;
        }
      }
    }

  }
  return num;
}


EffectLayout::EffectLayout(Effect *belongsToEffect) {
  m_belongsToEffect = belongsToEffect;
}

EffectLayout::EffectLayout() {
  m_belongsToEffect = NULL;
}

EffectLayout::~EffectLayout() {
}

int EffectLayout::readVec(std::string s, float *vec) {

  char delims[] = ",";
  char *result = NULL;
  char *strtok_context;
  char str[256];

  strcpy(str, s.c_str());

  result = strtok_s(str, delims, &strtok_context);
  int i=0;
  while (result != NULL) {
    if (sscanf(result, "%f", &vec[i]) > 0) {
      // vec[i] = 0.0f;
      i++;
    } 
    result = strtok_s(NULL, delims, &strtok_context);
  }
  return i;
}

void Trim(string *str) {
  while(str->length() && isspace(str->at(0)))
    str->erase(0, 1);

  while(str->length() && isspace(str->at(str->length() - 1)))
    str->erase(str->length() - 1, 1);
}



int EffectLayout::readParams(std::string s, std::vector<std::string> *params) {

  char delims[] = ",";
  char *result = NULL;
  char *strtok_context;
  char str[256];

  strcpy(str, s.c_str());

  result = strtok_s(str, delims, &strtok_context);
  int i=0;
  while (result != NULL) {
    std::string param = result;
    if (params) {
      Trim(&param);
      int n=param.find("$");
      if (n != std::string::npos) {
        param.replace(n, 1, "");
      }
      (*params).push_back(param);
    }
    i++;
    result = strtok_s(NULL, delims, &strtok_context);
  }
  return i;
}


bool EffectLayout::readFloat(std::string s, float *v) {
  if (sscanf(s.c_str(), "%f", v)<=0) {
    *v = 0.0f;
    return false;
  }
  return true;
}

bool EffectLayout::readInt(std::string s, int *v) {
  if (sscanf(s.c_str(), "%d", v)<=0) {
    *v = 0;
    return false;
  }
  return true;
}

// from timeline.cpp
extern void readEqualsStr(const char *str, const char *needle, char *result);

void EffectLayout::AddRecursiveStructure(const TiXmlNode *pn, EffectParam *lft, std::vector<std::string> *paramsIn, int level, std::string effectName) {

  while (pn) {

    const char *name = pn->Value();
    if (pn->Type() == TiXmlNode::COMMENT) {
      pn = pn->NextSibling();
      continue;
    }
    const TiXmlNode *txn = pn->FirstChild();

    EffectParam newLft;
    newLft.m_belongsToLayout = this;

    newLft.name = name;

    newLft.type = "";
    newLft.lang = "";

    const char* att;
    att = pn->ToElement()->Attribute("type");
    if (att) {
      newLft.type = att;
    }
    att = pn->ToElement()->Attribute("lang");
    if (att) {
      newLft.lang = att;
    }
    att = pn->ToElement()->Attribute("id");
    if (att) {
      newLft.id = att;
    }
    att = pn->ToElement()->Attribute("ref");
    if (att) {
      newLft.ref = att;
    }
    att = pn->ToElement()->Attribute("start");
    if (att) {
      float arvo;
      std::stringstream ss;
      ss << att;
      ss >> arvo;
      newLft.start = arvo;
    }
    att = pn->ToElement()->Attribute("end");
    if (att) {
      float arvo;
      std::stringstream ss;
      ss << att;
      ss >> arvo;
      newLft.end = arvo;
    }
    att = pn->ToElement()->Attribute("loopEnd");
    if (att) {
      float arvo;
      std::stringstream ss;
      ss << att;
      ss >> arvo;
      newLft.loopEnd = arvo;
    }    
    att = pn->ToElement()->Attribute("fadeIn");
    if (att) {
      float arvo;
      std::stringstream ss;
      ss << att;
      ss >> arvo;
      newLft.fadeIn = arvo;
    }    
    att = pn->ToElement()->Attribute("fadeOut");
    if (att) {
      float arvo;
      std::stringstream ss;
      ss << att;
      ss >> arvo;
      newLft.fadeOut = arvo;
    }    
    att = pn->ToElement()->Attribute("fadeInPow");
    if (att) {
      float arvo;
      std::stringstream ss;
      ss << att;
      ss >> arvo;
      newLft.fadeInPow = arvo;
    }    
    att = pn->ToElement()->Attribute("fadeOutPow");
    if (att) {
      float arvo;
      std::stringstream ss;
      ss << att;
      ss >> arvo;
      newLft.fadeOutPow = arvo;
    }    
    att = pn->ToElement()->Attribute("hide");
    if (att) {
      int arvo;
      std::stringstream ss;
      ss << att;
      ss >> arvo;
      newLft.hide = arvo;
    }   
    att = pn->ToElement()->Attribute("loop");
    if (att) {
      int arvo;
      std::stringstream ss;
      ss << att;
      ss >> arvo;
      newLft.loop = arvo;
    }   
    att = pn->ToElement()->Attribute("clear");
    if (att) {
      newLft.clear = att;
    }
    att = pn->ToElement()->Attribute("op");
    if (att) {
      newLft.op = att;
    }

    att = pn->ToElement()->Attribute("curve");
    if (att) {
      newLft.fadeInCurve = att;
      newLft.fadeOutCurve = att;
    }

    att = pn->ToElement()->Attribute("fadeInCurve");
    if (att) {
      newLft.fadeInCurve = att;
    }
    att = pn->ToElement()->Attribute("fadeOutCurve");
    if (att) {
      newLft.fadeOutCurve = att;
    }


    if (txn && (txn->FirstChild() || txn->Type() == TiXmlNode::ELEMENT)) {
      AddRecursiveStructure(txn, &newLft, paramsIn, level+1, effectName);
    } else {
      const char *elementValue = "";
      if (txn)
        elementValue = txn->Value();

      char strNow[512];
      char strElementName[512];
      char strElementDefault[512] = "0.0f";

      char paramName[512] = "";

      bool bHasDynamicParams = false;

      if (strstr(elementValue, "$")) {

        bHasDynamicParams = true; 
/*
        strcpy(strElementName, elementValue);
        if (char *eqPos = strstr(strElementName, "=")) {
          *eqPos = 0;
          strcpy(paramName, strElementName);
          strcpy(strElementDefault, eqPos+1);
        }
        bool found = false;
        if (paramsIn) {
          std::vector<std::string>::iterator it = paramsIn->begin();
          while (it != paramsIn->end()) {
            readEqualsStr((*it).c_str(), strElementName, strNow);
            if (strNow[0] != 0) {
              elementValue = strNow;
              found = true;
              break;
            }
            it++;
          }
        }
        if (!found) {
          if (strlen(strElementDefault) == 0) {
            char errMsg[1024];
            sprintf(errMsg, "Error:\nMissing \"%s\" from timeline params list for effect file \"%s\"", strstr(elementValue, "$")+1, this->layoutFile.c_str());
            MessageBox(NULL, errMsg, "Error in timeline", MB_ICONERROR);
            exit(-1);
          } else {
            // use the default value
            elementValue = strElementDefault;
          }
        }

        newLft.paramNames.push_back(paramName);
        if (strcmp(paramName, "") != 0) {
          std::string syncId = effectName+"."+paramName;
          const struct sync_track *pst = sync_get_track(GetRocket(), syncId.c_str());
          m_syncTracks[syncId] = pst;
        }
*/
        newLft.paramNames.clear();
        newLft.m_syncTracks.clear();
        int i = readParams(elementValue, &newLft.paramNames);
        newLft.numVec = i;
        if (newLft.paramNames.size()) {
          for (std::vector<std::string>::iterator itParam=newLft.paramNames.begin(); itParam!=newLft.paramNames.end(); itParam++) {
            std::string paramName = *itParam;
            std::string syncId = effectName+"."+paramName;
            if (strstr(paramName.c_str(), ".")) {
              syncId = paramName;
            }
            if (strstr(paramName.c_str(), "g_")==paramName.c_str()) {
              newLft.m_globalParam = paramName;
            } else {
              const struct sync_track *pst = sync_get_track(GetRocket(), syncId.c_str(), g_syncEditor);
              newLft.m_syncTracks.push_back(pst);
            }
          }
        }
      }
      if (!bHasDynamicParams) {
        int i = readVec(elementValue, newLft.valueVec);
        if (i==0) {
          newLft.valueStr = elementValue;
        }
        newLft.numVec = i;
      }
    }

    if (lft) {
      lft->params.push_back(newLft);
    }/* else {
      le->params.push_back(newLft);
    }*/

    pn = pn->NextSibling();
    if (level == 0)
      break;
  }
}

int EffectLayout::readFromFile(std::string rootDir, std::string xmlFile, std::vector<std::string> *paramsIn) {
  char buf[512];

//  this->elements.clear();
//  this->params.clear();

  InsertLoadedFileEffectLayout(xmlFile);

  m_paramRoot.m_belongsToLayout = this;

  m_rootDir = rootDir;

  xmlFile = rootDir+xmlFile;

  this->layoutFile = xmlFile;

  std::string effectName = xmlFile;
  char effectNameStr[512];
  char effectNameCopyStr[512];
  strcpy(effectNameCopyStr, effectName.c_str());

  int slashPos=0;
  for (int i=0; i<strlen(effectNameCopyStr); i++) {
    if (effectNameCopyStr[i] == '/') {
      slashPos = i+1;
      break;
    }
  }
  strcpy(effectNameStr, effectNameCopyStr+slashPos);
  effectName = effectNameStr;
  int xmlPos;
  xmlPos = effectName.find(".xml");
  if (xmlPos != std::string::npos) {
    effectName.replace(xmlPos, 4, "");
  }
  xmlPos = effectName.find("$");
  if (xmlPos != std::string::npos) {
    effectName.replace(xmlPos, 1, "");
  }
  m_effectName = effectName;

  if (m_belongsToEffect) {
    m_belongsToEffect->SetEnableSyncVariable(m_effectName+".ON");
  }

  TiXmlDocument doc;
  doc.LoadFile(xmlFile.c_str());
  if (doc.Error()) {
    sprintf(buf, "Error parsing layout %s: %s\n", xmlFile.c_str(), doc.ErrorDesc());
    OutputDebugString(buf);
    return 0;
  } else {
    sprintf(buf, "Reading layout %s\n", xmlFile.c_str());
    OutputDebugString(buf);
    TiXmlHandle hDoc(&doc);
    TiXmlElement* pElem;
    TiXmlHandle hRoot(0);

    pElem=hDoc.FirstChildElement().Element();	
    if (pElem) {
      const char *effectType = pElem->Attribute("type");
      if (effectType) {
        m_effectType = effectType;
      }
      hRoot=TiXmlHandle(pElem);

      TiXmlNode *pn;
      TiXmlNode *nodeNow;
      TiXmlHandle fc = hRoot.FirstChild();
      nodeNow = fc.ToNode();
      if (nodeNow) {
        do {
          pn = nodeNow;
          if (pn) {
           // this->params.push_back(ep);
            AddRecursiveStructure(pn, &m_paramRoot, paramsIn, 0, effectName);
          } else {
            sprintf(buf, "Error parsing layout: no element\n");
            OutputDebugString(buf);
          }
          nodeNow = nodeNow->NextSibling();
        } while (nodeNow);
      }
    }
  }

  return 1;
}

const string EffectParam::getString(std::string name) const {
 // if (getN(name)==0)
 //   return "";

  return get(name)->valueStr;
}

const float EffectParam::getFloat(string paramName, float defaultValue) const {
  D3DXVECTOR4 dV4 = D3DXVECTOR4(defaultValue, 0.0, 0.0, 1.0f);
  D3DXVECTOR4 r = getVec4(paramName, 0, &dV4);
  return r.x;
}

const int EffectParam::getInt(string paramName) const {
  D3DXVECTOR4 r = getVec4(paramName);
  return (int)(r.x + ((r.x < 0) ? -0.5 : 0.5));
}


const D3DXVECTOR3 EffectParam::getVec3(string paramName, float *pGlobalFade) const {
  D3DXVECTOR4 r = getVec4(paramName, pGlobalFade);
  return D3DXVECTOR3(r.x, r.y, r.z);
}


float GetFadeIn(float t, float fadeTime, float start, bool cubic, float pow) {
  if (t >= start) {
    t-=start;
    if (fadeTime > 0.0f) {
      t/=fadeTime;
      if (t > 1.0f)
        t = 1.0f;
      t = t;
    } else {
      t = 1.0f;
    }
  } else {
    t = 0.0f;
  }
  if (cubic) {
    t = (3.0f*t*t-2.0f*t*t*t);
  }
  return powfsafe(t, pow);
}

float GetFadeOut(float t, float fadeTime, float end, bool cubic, float pow) {
  if (t >= (end-fadeTime)) {
    t-=(end-fadeTime);
    if (fadeTime > 0.0f) {
      t/=fadeTime;
      if (t > 1.0f)
        t = 1.0f;
      t = 1.0f-t;
    } else {
      t = 0.0f;
    }
  } else {
    t = 1.0f;
  }
  if (cubic) {
 //   t = (2.0f*t*t*t-3.0f*t*t+1.0f);
    t = (3.0f*t*t-2.0f*t*t*t);

  }
  return powfsafe(t, pow);
}



const void EffectParam::GetFadeParams(float *fade, int *isOn, float *modTime, float *startTime, float *endTime, float freq) const {
  *startTime = start;
  *endTime = end;
  float loopEndTime = loopEnd;
  int loops = loop;

  Effect *pEf = m_belongsToLayout->m_belongsToEffect;
  if (!pEf)
    pEf = GetCurrentEffect();


  if (*startTime < -500.0f)
    *startTime = 0.0f;
  if (*endTime < -500.0f) {
    
    *endTime = pEf->GetEndTime() - pEf->GetStartTime();
  }
  if (loopEndTime < -500.0f)
    loopEndTime = *endTime;

  int iTime = (int)((pEf->GetEffectTime()*freq-*startTime)*100000.0f);
  *modTime = (iTime % (int)(loopEndTime*100000.0f))/100000.0f;
  int loopingAtRound = iTime/((int)(loopEndTime*100000.0f));

  (*isOn) = 0;

  if (iTime >= 0 && (*modTime) >= 0.0f && (*modTime) < (*endTime-*startTime) && (loops<0 || loopingAtRound <= loops)) {

    (*isOn) = 1;

    bool cubicFadeIn = false;
    bool cubicFadeOut = false;

    if (fadeInCurve == "cubic")
      cubicFadeIn = true;

    if (fadeOutCurve == "cubic")
      cubicFadeOut = true;
    
    *fade = 1.0f;
    
    *fade *= GetFadeIn(*modTime, fadeIn, 0.0f, cubicFadeIn, fadeInPow);
    *fade *= GetFadeOut(*modTime, fadeOut, (*endTime-*startTime), cubicFadeOut, fadeOutPow);

  } else {
    *fade = 0.0f;
  }
}


extern double fmod_get_row();

const float EffectParam::GetDynamicValue(int i) const {
  if (i < m_syncTracks.size()) {
    float v = sync_get_val(m_syncTracks[i], fmod_get_row());
    return v;
  } else {
    return valueVec[i];
  }
}

std::map<std::string, Curve*> globalAdks;

Curve *GetGlobalAdk(std::string fileName) {
  std::map<std::string, Curve*>::iterator it=globalAdks.find(fileName);
  if (it!=globalAdks.end()) {
    return (*it).second;
  }
  Curve *pCurve = new Curve();
  pCurve->LoadCurve(fileName.c_str());
  globalAdks[fileName] = pCurve;
  return pCurve;
}


std::map<std::string, D3DXVECTOR4> g_globalFloatParams;


void SetGlobalFloatParam(const char *s, float p) {
  g_globalFloatParams[s] = D3DXVECTOR4(p, 0.0f, 0.0f, 0.0f);
}

void SetGlobalFloatParamVec4(const char *s, D3DXVECTOR4 *pParam) {
  g_globalFloatParams[s] = *pParam;
}

D3DXVECTOR4 GetFromGlobalFloatParams(const std::string &s) {
  D3DXVECTOR4 res(0.0f, 0.0f, 0.0f, 0.0f);
  std::map<std::string, D3DXVECTOR4>::iterator it = g_globalFloatParams.find(s);
  if (it==g_globalFloatParams.end()) {
    return res;
  }
  res = (*it).second;
  return res;
}


const D3DXVECTOR4 EffectParam::getVec4(string paramName, float *pGlobalFade, D3DXVECTOR4 *pDefaultValue) const {

  D3DXVECTOR4 result;
   
  if (pDefaultValue) {
    result = *pDefaultValue;
  } else {
    result = D3DXVECTOR4(0.0f, 0.0f, 0.0f, 1.0f);
  }

  float valueVecDynamic[4];

  valueVecDynamic[0] = valueVec[0];
  valueVecDynamic[1] = valueVec[1];
  valueVecDynamic[2] = valueVec[2];
  valueVecDynamic[3] = valueVec[3];

  if (m_syncTracks.size()) {
    int i=0;
    for (std::vector<const sync_track*>::const_iterator it=m_syncTracks.begin(); it!=m_syncTracks.end(); it++) {
      if (i<4) {
        float v = sync_get_val((*it), fmod_get_row());
        valueVecDynamic[i] = v;
      }
      i++;
    }
    if (i!=0) { // control from sync tracks -> ignore the default value, otherwise it would incorrectly sum up 
      result = D3DXVECTOR4(0.0f, 0.0f, 0.0f, 1.0f); 
    }
  }

  if (!m_globalParam.empty()) {
    result = GetFromGlobalFloatParams(m_globalParam);
  }

  if (pGlobalFade)
    *pGlobalFade = 1.0f;

  if (strcmp(paramName.c_str(),"")==0) {
    // TBD: lisaa tahan tarkastus onko talla parametrilla <float>, <curve> tms alielementtia
    result = D3DXVECTOR4(valueVecDynamic[0], valueVecDynamic[1], valueVecDynamic[2], valueVecDynamic[3]);
    return result;
  }

  Effect *pEf = m_belongsToLayout->m_belongsToEffect;
  if (!pEf)
    pEf = GetCurrentEffect();


  int numE = getN(paramName);
  for (int i=0; i<numE; i++) {
    const EffectParam *ep = get(paramName, i);

  //  valueSet = true;

    float fade;
    int isOn;
    float modTime;
    float startTime;
    float endTime;

    ep->GetFadeParams(&fade, &isOn, &modTime, &startTime, &endTime);

    if (ep->m_syncTracks.size() && i==0) { // control from sync tracks -> ignore the default value, otherwise it would incorrectly sum up 
      result = D3DXVECTOR4(0.0f, 0.0f, 0.0f, 1.0f);
    }

    if (isOn) {

      float globalFade = 1.0f;

      if (ep->clear == "1")
        globalFade = (1.0f-fade);

      result *= globalFade;

      if (pGlobalFade)
        *pGlobalFade = globalFade;

      D3DXVECTOR4 resultte = D3DXVECTOR4(0.0f, 0.0f, 0.0f, 0.0f);
      bool resultteSet = false;

      if (!ep->m_globalParam.empty()) {
        resultte = GetFromGlobalFloatParams(ep->m_globalParam);
        resultteSet = true;
      } else if (ep->numVec) {
        resultte = D3DXVECTOR4(ep->GetDynamicValue(0), ep->GetDynamicValue(1), ep->GetDynamicValue(2), ep->GetDynamicValue(3))*fade;
        resultteSet = true;
      } else {
        // curve?
        const EffectParam *epc = ep->get("curve");

        float freq;
        
        if (epc->getN("freq"))
          freq = epc->getFloat("freq");
        else
          freq = 1.0f;

        float timeStep = pEf->GetTimeStep();


        D3DXVECTOR3 scale;
        if (epc->getN("scale"))
          scale = epc->getVec3("scale");
        else
          scale = D3DXVECTOR3(1.0f, 1.0f, 1.0f);

        D3DXVECTOR3 position;
        if (epc->getN("position")) {
          position = epc->getVec3("position");
        }

        D3DXVECTOR3 rotate;
        if (epc->getN("rotate")) {
          rotate = epc->getVec3("rotate");
        }

        float time_offset = 0.0f;


        if (epc->getN("time_offset")) {
          time_offset = epc->getFloat("time_offset");
        }

        if (epc->getN("adk")) {
          std::string adkFileName = epc->getString("adk");
          Curve *pCurve = GetGlobalAdk(adkFileName);
          if (pCurve) {
            D3DXVECTOR3 pp1 = pCurve->GetPos(modTime*freq+time_offset);
            resultte = D3DXVECTOR4(pp1.x*scale.x, pp1.y*scale.y, pp1.z*scale.z, 1.0f)*fade;
            resultteSet = true;
          }
        } else {
          const EffectParam *epcPoints = epc->get("points");
          int pointNum = epcPoints->getN("point");
          if (pointNum) {
            Curve curve;
            float ctd=(endTime-startTime)/pointNum;
            float ct=0.0f; // startTime;
            for (int pi=0; pi<pointNum; pi++) {
              D3DXVECTOR4 v4 = epcPoints->get("point", pi)->getVec4("");
              curve.AddPoint(ct, &D3DXVECTOR3(v4.x, v4.y, v4.z));
              ct += ctd;
            }
            float loopEnd = endTime;
            if (epc->loopEnd > -500.0f)
              loopEnd = epc->loopEnd;
            curve.SetLoop(epc->loop, loopEnd);
            D3DXVECTOR3 pp1 = curve.GetPos(modTime*freq+time_offset);
            resultte = D3DXVECTOR4(pp1.x*scale.x, pp1.y*scale.y, pp1.z*scale.z, 1.0f)*fade;
            resultteSet = true;
          }
          pointNum = epcPoints->getN("pointKey");
          if (pointNum) {
            Curve curve;
            
            for (int pi=0; pi<pointNum; pi++) {
              D3DXVECTOR4 v4 = epcPoints->get("pointKey", pi)->getVec4("");
              curve.AddPoint(startTime*0.0f+v4.x, &D3DXVECTOR3(v4.y, v4.z, v4.w));
            }
            float loopEnd = endTime;
            if (epc->loopEnd > -500.0f)
              loopEnd = epc->loopEnd;
            curve.SetLoop(epc->loop, loopEnd);
            float tdd2 = 0.0001f;
            D3DXVECTOR3 pp1 = curve.GetPos(modTime*freq+time_offset);
            /*
            D3DXVECTOR3 pp2 = curve.GetPos(modTime*freq+time_offset+tdd2);
            pp2 = pp2-pp1;
            float pp2Len = D3DXVec3Length(&pp2);
            float nopsu = pp2Len/tdd2;
            char kukko[256];
            sprintf(kukko, "nopsu %f\n", nopsu);
            OutputDebugString(kukko);
            */
            
            resultte = D3DXVECTOR4(pp1.x*scale.x, pp1.y*scale.y, pp1.z*scale.z, 1.0)*fade;
            resultteSet = true;
          }

            /*
            pointNum = 50;

            srand(123);
            float koho = 0.0f;
            for (int pi=0; pi<pointNum; pi++) {
              curve.AddPoint(start+koho, &D3DXVECTOR3(((float)rand()/RAND_MAX-0.5f)*130.0f, ((float)rand()/RAND_MAX-0.5f)*130.0f, ((float)rand()/RAND_MAX-0.5f)*30.0f-10.0f));
              float kuu = (float)rand()/RAND_MAX;
              koho += (0.50f+kuu*kuu*2.0f);
            }
            */

          int fileNum = epcPoints->getN("file");
          if (fileNum) {
  //          <file>rata1.x</file>
  //          <track>cube</track>
  //          <get>position</get>
            std::string fileName = epcPoints->getString("file");
            std::string trackName = epcPoints->getString("track");
            std::string trackType = epcPoints->getString("get");
            int meshIndex = g_D3DApp->addAnimatedMesh(fileName.c_str(), NULL, NULL);
            Mesh *mesh = g_D3DApp->getMesh(meshIndex);
            if (mesh) {
              D3DXMATRIXA16 matWorld;
              D3DXMatrixIdentity(&matWorld);
              mesh->Update(modTime*freq+time_offset);
              g_D3DApp->getAnimatedMeshMatrix(*mesh, trackName, &matWorld, NULL);
              D3DXVECTOR3 oScale;
              D3DXQUATERNION oRot;
              D3DXVECTOR3 oTrans;

              D3DXMatrixDecompose(&oScale, &oRot, &oTrans, &matWorld);
              
              if (_stricmp(trackType.c_str(), "position")==0) {
                resultte = D3DXVECTOR4(oTrans.x*scale.x, oTrans.y*scale.y, oTrans.z*scale.z, 1.0)*fade;
                resultteSet = true;
              }
              if (_stricmp(trackType.c_str(), "scale")==0) {
                resultte = D3DXVECTOR4(oScale.x*scale.x, oScale.y*scale.y, oScale.z*scale.z, 1.0)*fade;
                resultteSet = true;
              }
              if (_stricmp(trackType.c_str(), "rotate")==0) {
                D3DXVECTOR3 oRotEuler;
             //   D3DXGetY
  // TBD: find out how to get roll, pitch and yaw from the quaternion rotation
                resultte = D3DXVECTOR4(oRotEuler.x, oRotEuler.y, oRotEuler.z, 1.0)*fade;
                resultteSet = true;
              }
            }
          }
        }
      }

      if (resultteSet) {
        if (ep->op == "+") {
          result += resultte;
        } else if (ep->op == "*") {
          result.x = result.x*resultte.x;
          result.y = result.y*resultte.y;
          result.z = result.z*resultte.z;
          result.w = result.w*resultte.w;
        } else if (ep->op == "-") {
          result -= resultte;
        } else if (ep->op == "pow") {
          result.x = powfsafe(result.x, resultte.x);
          result.y = powfsafe(result.y, resultte.y);
          result.z = powfsafe(result.z, resultte.z);
          result.w = powfsafe(result.w, resultte.w);
        } else if (ep->op == "") {
          result = resultte;
        }
      }
    }
  }

  return result;
}