#pragma once

#include "glfunctions.h"

#include "../shaders/output/intogbuffer.vert.h"
#include "../shaders/output/intogbuffer.frag.h"
#include "../shaders/output/postfx.vert.h"
#include "../shaders/output/postfx.frag.h"
#include "../shaders/output/blur.frag.h"


#define kResolutionX 1920
#define kResolutionY 1080
#define kShadowMapSize 2048

GLuint gRenderGbufferFBO;

// these need to be 1..4, so that blur can just use that loop
#define kGLTexAlbedo 1
#define kGLTexNormals 2
#define kGLTexDepth 3
#define kGLTexShadowmap 4


#define kGLTexBlurTempColor 5
#define kGLTexBlurTempDepth 6

GLuint gRenderShadowFBO;

GLuint gRenderBlurFBO;


struct Shader
{
	GLuint program;
#define DO_UNIFORM(name) int name;
#include "uniformslist.h"
#undef DO_UNIFORM
};

Shader gShaderPostfx;
Shader gShaderGbuffer;
Shader gShaderBlur;

static void BindGLTexture(int stage, GLuint tex)
{
	GL_CALL(gl.activeTexture(GL_TEXTURE0+stage));
	GL_CALL(glBindTexture(GL_TEXTURE_2D, tex));
}

static void CreateSizedTexture(GLuint tex, GLenum format, GLenum internalFormat, GLenum compType, int width, int height)
{
	glBindTexture(GL_TEXTURE_2D, tex);
	glTexImage2D(GL_TEXTURE_2D, 0, internalFormat, width, height, 0, format, compType, NULL);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_CLAMP_TO_EDGE);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_CLAMP_TO_EDGE);
}

static void CreateScreenTexture(GLuint tex, GLenum format, GLenum internalFormat, GLenum compType)
{
	CreateSizedTexture(tex, format, internalFormat, compType, kResolutionX, kResolutionY);
}

static void CreateScreenColorTexture(GLuint tex)
{
	CreateScreenTexture(tex, GL_RGBA, GL_RGBA16F, GL_HALF_FLOAT);
}

static void CreateScreenDepthTexture(GLuint tex)
{
	CreateScreenTexture(tex, GL_DEPTH_COMPONENT, GL_DEPTH_COMPONENT, GL_UNSIGNED_BYTE);
}

static void CreateShadowmap(GLuint tex, int size)
{
	CreateSizedTexture(tex, GL_DEPTH_COMPONENT, GL_DEPTH_COMPONENT, GL_UNSIGNED_BYTE, kShadowMapSize, kShadowMapSize);
}

static GLuint CreateFBO()
{
	GLuint fbo;
	gl.genFramebuffers(1, &fbo);
	gl.bindFramebuffer(GL_FRAMEBUFFER, fbo);
	return fbo;
}

static void AttachTextureToFBO(GLenum attachment, GLuint tex)
{
	gl.framebufferTexture(GL_FRAMEBUFFER, attachment, tex, 0);
}

static void CheckFBO()
{
#if !FINALBUILD
	if (gl.checkFramebufferStatus(GL_FRAMEBUFFER) != GL_FRAMEBUFFER_COMPLETE)
	{
		DebugBreak();
	}
#endif
}

static void AddShaderStage(GLuint prog, char stage, const char* source)
{
	GLuint s = gl.createShader(GL_VERTEX_SHADER-stage);
	GL_CALL(gl.shaderSource(s, 1, &source, NULL));
	GL_CALL(gl.compileShader(s));
#if !FINALBUILD
	int success = 0;
	char infoLog[1024];
	gl.getShaderiv(s, GL_COMPILE_STATUS, &success);
	if (!success)
	{
		gl.getShaderInfoLog(s, sizeof(infoLog), NULL, infoLog);
		DebugBreak();
	}
#endif
	GL_CALL(gl.attachShader(prog, s));
}

static const char* kUniformNames[] = {
#define DO_UNIFORM(name) #name,
#include "uniformslist.h"
#undef DO_UNIFORM
};


static void CreateShaderProgram(const char* vsSource, const char* fsSource, Shader& shader)
{
	GLuint prog = gl.createProgram();
	AddShaderStage(prog, 0, vsSource);
	AddShaderStage(prog, 1, fsSource);
	GL_CALL(gl.linkProgram(prog));
#if !FINALBUILD
	int success = 0;
	char infoLog[1024];
	gl.getProgramiv(prog, GL_LINK_STATUS, &success);
	if (!success)
	{
		gl.getProgramInfoLog(prog, sizeof(infoLog), NULL, infoLog);
		DebugBreak();
	}
#endif
	shader.program = prog;
	// note: this assumes that Shader struct has just uniform index members right after "program"
	int* ptr = (int*)(&shader.program) + 1;
	for (int i = 0; i < sizeof(kUniformNames) / sizeof(kUniformNames[0]); ++i)
	{
		*ptr++ = gl.getUniformLocation(prog, kUniformNames[i]);
	}
}


static void RenderCreateResources()
{
	gRenderGbufferFBO = CreateFBO();
	CreateScreenColorTexture(kGLTexAlbedo); AttachTextureToFBO(GL_COLOR_ATTACHMENT0, kGLTexAlbedo);
	CreateScreenColorTexture(kGLTexNormals); AttachTextureToFBO(GL_COLOR_ATTACHMENT1, kGLTexNormals);
	CreateScreenDepthTexture(kGLTexDepth); AttachTextureToFBO(GL_DEPTH_ATTACHMENT, kGLTexDepth);
	CheckFBO();

	gRenderShadowFBO = CreateFBO();
	CreateShadowmap(kGLTexShadowmap, kShadowMapSize); AttachTextureToFBO(GL_DEPTH_ATTACHMENT, kGLTexShadowmap);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_COMPARE_MODE, GL_COMPARE_R_TO_TEXTURE);
	CheckFBO();

	gRenderBlurFBO = CreateFBO();
	CreateScreenColorTexture(kGLTexBlurTempColor);
	CreateScreenDepthTexture(kGLTexBlurTempDepth);

	CreateShaderProgram(postfx_vert, blur_frag, gShaderBlur);

	CreateShaderProgram(intogbuffer_vert, intogbuffer_frag, gShaderGbuffer);


	CreateShaderProgram(postfx_vert, postfx_frag, gShaderPostfx);
	gl.useProgram(gShaderPostfx.program);
	GL_CALL(gl.uniform1i(gShaderPostfx.tA, 0));
	GL_CALL(gl.uniform1i(gShaderPostfx.tN, 1));
	GL_CALL(gl.uniform1i(gShaderPostfx.tD, 2));
	GL_CALL(gl.uniform1i(gShaderPostfx.tS, 3));
	gl.useProgram(0);
}

static float BlurIteration(float val, float valz, GLuint tex, GLuint dest, bool depth)
{
	GL_CALL(gl.uniform4f(gShaderBlur.of, val/kResolutionX, val/kResolutionY, valz, 0));
	AttachTextureToFBO(depth ? GL_COLOR_ATTACHMENT0 : GL_DEPTH_ATTACHMENT, 0);
	AttachTextureToFBO(depth ? GL_DEPTH_ATTACHMENT : GL_COLOR_ATTACHMENT0, dest);
	CheckFBO();
	if (depth)
		glClear(GL_DEPTH_BUFFER_BIT);
	GL_CALL(glViewport(0, 0, kResolutionX, kResolutionY));
	BindGLTexture(0, tex);
	GL_CALL(glDrawArrays(GL_TRIANGLES, 0, 3));
	return val + 1;
}

static void BlurTexture(GLuint tex, GLuint tempTex, int iterations, bool normals, bool depth)
{
	gl.useProgram(gShaderBlur.program);
	GL_CALL(gl.uniform1i(gShaderBlur.tA, 0));

	float val = 0.5f;
	float valz = normals ? 1 : 0;
	for (int i = 0; i < iterations; ++i)
	{
		GL_CALL(gl.bindFramebuffer(GL_FRAMEBUFFER, gRenderBlurFBO));
		val = BlurIteration(val, valz, tex, tempTex, depth);
		val = BlurIteration(val, valz, tempTex, tex, depth);
	}

	gl.useProgram(0);
}
