
// --------------------------------------------------------------------------
//  Synth.cpp - Softsynth.
//  Copyright (c) 2001. Remage / Fresh!mindworkz.
// --------------------------------------------------------------------------

/*
Waveshape distortion:
    x = input in [-1..1]
    y = output
    k = 2*amount/(1-amount);
    amount = [-1..1[
    f(x) = (1+k)*x/(1+k*abs(x))
*/

#include <StdIO.h>
#include <StdLib.h>
#include <Windows.h>

#include "Fmath.h"
#include "Rand.h"

float Fpow( float Value, float Exp )
  {
    if ( Value == 0.0f ) 
      return 0.0f;
    if ( Value > 0.0f ) 
      return Fexp( Exp * Fln( Value )); else
      return - Fexp( Exp * Fln( -Value ));
   }

#include "Synth.h"

// --------------------------------------------------------------------------
//  Synth_BufferInitialize()
// --------------------------------------------------------------------------

SYNTH_BUFFER *Synth_BufferInitialize( float Len )
  {
    SYNTH_BUFFER *Buf;
    
    Buf  = (SYNTH_BUFFER*) HeapAlloc( GetProcessHeap(), HEAP_ZERO_MEMORY, sizeof( SYNTH_BUFFER ));
    Buf->Length = Fround( Len * SAMPLERATE );
    Buf->Ptr = (float*) HeapAlloc( GetProcessHeap(), HEAP_ZERO_MEMORY, 2 * sizeof(float) * Buf->Length );
    return Buf;
   }

// --------------------------------------------------------------------------
//  Synth_BufferCopy( SrcBuf )
// --------------------------------------------------------------------------

SYNTH_BUFFER *Synth_BufferCopy( SYNTH_BUFFER *SrcBuf )
  {
    unsigned int I;
    SYNTH_BUFFER *Buf;
    
    Buf  = (SYNTH_BUFFER*) HeapAlloc( GetProcessHeap(), HEAP_ZERO_MEMORY, sizeof( SYNTH_BUFFER ));
    Buf->Length = SrcBuf->Length;
    Buf->Ptr = (float*) HeapAlloc( GetProcessHeap(), HEAP_ZERO_MEMORY, 2 * sizeof(float) * Buf->Length );
    for ( I=0; I<2*Buf->Length; I++ )
      Buf->Ptr[I] = SrcBuf->Ptr[I];

    return Buf;
   }

// --------------------------------------------------------------------------
//  Synth_BufferCleanup()
// --------------------------------------------------------------------------

void Synth_BufferCleanup( SYNTH_BUFFER *Buf )
  {
    HeapFree( GetProcessHeap(), 0, (void*) Buf->Ptr );
    HeapFree( GetProcessHeap(), 0, (void*) Buf );
    Buf = NULL;
   }

// --------------------------------------------------------------------------
//  Synth_Amplify()
// --------------------------------------------------------------------------

void Synth_Amplify( SYNTH_BUFFER *Buf, float Ampl )
  {
    unsigned int I;
    float Val;

    for ( I=0; I<2*Buf->Length; I++ )
      {
        Val = Buf->Ptr[ I ] * Ampl;
        if ( Val > 1.0f ) Val = 1.0f;
        if ( Val < -1.0f ) Val = -1.0f;
        Buf->Ptr[ I ] = Val;
       }
   }

// --------------------------------------------------------------------------
//  Synth_Amplify()
// --------------------------------------------------------------------------

void Synth_Normalize( SYNTH_BUFFER *Buf )
  {
    unsigned int I;
    float Max = 0.0f;

    for ( I=0; I<2*Buf->Length; I++ )
      if ( Fabs( Buf->Ptr[ I ] ) > Max ) 
        Max = Fabs( Buf->Ptr[ I ] );

    Max = 1.0f / Max;

    for ( I=0; I<2*Buf->Length; I++ )
      Buf->Ptr[ I ] *= Max;
   }


// --------------------------------------------------------------------------
//  Synth_Interpolate_Spline()
// --------------------------------------------------------------------------

float Synth_Interpolate_Spline( float F0, float F1, float F2, float F3, float Time )
  {
    float HermitA, HermitB, HermitC;
    if (( Time < 0.0f ) || ( Time >= 1.0f )) return 0.0f;
    HermitA = ( 3 * ( F1 - F2 ) - F0 + F3 ) / 2;
    HermitB = 2 * F2 + F0 - ( 5 * F1 + F3 ) / 2;
    HermitC = ( F2 - F0 ) / 2;
    return F1 + (( HermitA * Time + HermitB ) * Time + HermitC ) * Time;
   }

// --------------------------------------------------------------------------
//  float Synth_BufferValue( Buf, Time, Loop )  
// --------------------------------------------------------------------------

float Synth_BufferValue( SYNTH_BUFFER *Buf, float Pos, char Flags )
  {
    int T;
    float Val0 = 0.0f, Val1 = 0.0f, Val2 = 0.0f, Val3 = 0.0f;

    T = (int) Fround( Ftrunc( Pos ));

    if (( T-2 >= 0 ) && ( T-1 < (int) Buf->Length )) Val0 = Buf->Ptr[ (T+1)*2 + (Flags&1) ];
    if (( T >= 0 ) && ( T < (int) Buf->Length )) Val1 = Buf->Ptr[ T*2 + (Flags&1) ];
    if (( T+1 >= 0 ) && ( T+1 < (int) Buf->Length )) Val2 = Buf->Ptr[ (T+1)*2 + (Flags&1) ];
    if (( T+2 >= 0 ) && ( T+2 < (int) Buf->Length )) Val3 = Buf->Ptr[ (T+2)*2 + (Flags&1) ];
    
    return Synth_Interpolate_Spline( Val0, Val1, Val2, Val3, Frac( Pos ));
   }

// --------------------------------------------------------------------------
//  Synth_Control()
// --------------------------------------------------------------------------

float Synth_ControlValue( SYNTH_CONTROL *Ctrl, float Time )
  {
    int I=1;
    float Time2, Value = 0.0f;

    switch ( Ctrl->Type )
      {
        case SYNTH_CTRL_FLOAT:
          Value = Ctrl->Values[ 0 ];
          break;
        case SYNTH_CTRL_ENV_LINEAR:
          while ( Ctrl->Values[ I*2 ] <=  Time ) I++;
          Time2 = ( Time - Ctrl->Values[ (I-1)*2 ] ) / ( Ctrl->Values[ I*2 ] - Ctrl->Values[ (I-1)*2 ] );
          Value =  Ctrl->Values[ (I-1)*2+1 ] + ( Ctrl->Values[ I*2+1 ] - Ctrl->Values[ (I-1)*2+1 ] ) * Time2;
          break;
        case SYNTH_CTRL_ENV_SPLINE:
          while ( Ctrl->Values[ I*2 ] <= Time ) I++;
          Time2 = ( Time - Ctrl->Values[ (I-1)*2 ] ) / ( Ctrl->Values[ I*2 ] - Ctrl->Values[ (I-1)*2 ] );
          Value = Synth_Interpolate_Spline( Ctrl->Values[ (I-2)*2+1 ], Ctrl->Values[ (I-1)*2+1 ], Ctrl->Values[ I*2+1 ], Ctrl->Values[ (I+1)*2+1 ], Time2 );
          break;
        case SYNTH_CTRL_ENV_LOG:
          if ( Time < Ctrl->Values[ 2 ] ) Value = Ctrl->Values[ 0 ]; else
          if ( Time > Ctrl->Values[ 3 ] ) Value = Ctrl->Values[ 1 ]; else
          Value = Ctrl->Values[ 0 ] + ( Ctrl->Values[ 1 ] - Ctrl->Values[ 0 ] ) * Fpow(( Time - Ctrl->Values[ 2 ] ) / ( Ctrl->Values[ 3 ] - Ctrl->Values[ 2 ] ), Ctrl->Values[ 4 ] );
          break;
        case SYNTH_CTRL_LFO_SINE:
          Value = Ctrl->Values[ 0 ] + Ctrl->Values[ 1 ] * Fsin( Time * Ctrl->Values[ 2 ] * Fmath_2Pi );
          break;
       }

    return Value;
   }

// --------------------------------------------------------------------------
//  Synth_GenerateWave()
// --------------------------------------------------------------------------

void Synth_GenerateWave( SYNTH_BUFFER *Buf, char Wave, unsigned char NoiseFreq, float NoiseAmpl, SYNTH_CONTROL *FreqCtrl, SYNTH_CONTROL *DistCtrl, SYNTH_CONTROL *AmplCtrl, SYNTH_CONTROL *PanCtrl )
  {
    unsigned int I, Noise = 0;
    float Time, Dist, Ampl, Pan, Value = 0.0f, Freq = 0.0f;
    
    for ( I=0; I<Buf->Length; I++ )
      {
        Time = (float) I / (float) SAMPLERATE;
        Ampl = Synth_ControlValue( AmplCtrl, Time );
        Dist = Synth_ControlValue( DistCtrl, Time );

        switch ( Wave )
          {
            case SYNTH_WAVE_SINE:
              Value = Ampl * Fsin( Fpow( Frac( Freq ), Dist ) * Fmath_2Pi );
              break;
            case SYNTH_WAVE_PULSE:
              Value = Ampl * (( Fsin( Fpow( Frac( Freq ), Dist ) * Fmath_2Pi ) > 0.0f ) ? 1.0f : -1.0f );
              break;
            case SYNTH_WAVE_SAW:
              Value = Ampl * ( 2.0f * Fpow( Frac( Freq ), Dist ) - 1.0f );
              break;
            case SYNTH_WAVE_TRIANGLE:
              Value = 4.0f * Fpow( Frac( Freq ), Dist );
              if ( Value > 1.0f ) Value = 2.0f - Value;
              if ( Value < -1.0f ) Value = -2.0f - Value;
              Value *= Ampl;
              break;
           }

        Pan = Synth_ControlValue( PanCtrl, Time );
        Buf->Ptr[ 2*I+0 ] += ( 1.0f - Pan ) * Value;
        Buf->Ptr[ 2*I+1 ] += Pan * Value;

        Freq += Synth_ControlValue( FreqCtrl, Time ) / SAMPLERATE;
        if ( NoiseFreq )
          if ( ++Noise == NoiseFreq )
            {
              Noise = 0;
              Freq += NoiseAmpl * Fsin( Fmath_2Pi * ((float) ( Rand() & 0x1FFF ) / (float) 0x1FFF ));
             }
       }
   }

// --------------------------------------------------------------------------
//  Synth_DistortWave()
// --------------------------------------------------------------------------

void Synth_DistortWave( SYNTH_BUFFER *Buf, SYNTH_CONTROL *DistCtrl, SYNTH_CONTROL *DryWetCtrl )
  {
    unsigned int I;
    float Time, Dist, Dry, Wet;

    for ( I=0; I<Buf->Length; I++ )
      {
        Time = (float) I / (float) SAMPLERATE;
        Dist = Synth_ControlValue( DistCtrl, Time );
        Wet = Synth_ControlValue( DryWetCtrl, Time );
        Dry = 1.0f - Wet;
        Buf->Ptr[ 2*I+0 ] = Buf->Ptr[ 2*I+0 ] * Dry + Fpow( Buf->Ptr[ 2*I+0 ], Dist ) * Wet;
        Buf->Ptr[ 2*I+1 ] = Buf->Ptr[ 2*I+1 ] * Dry + Fpow( Buf->Ptr[ 2*I+1 ], Dist ) * Wet;
       }
   }

// --------------------------------------------------------------------------
//  Synth_FilterWave()
// --------------------------------------------------------------------------

void Synth_FilterWave( SYNTH_BUFFER *Buf, char Type, SYNTH_CONTROL *Cutoff, SYNTH_CONTROL *Resonance, SYNTH_CONTROL *DryWetCtrl )
  {
    float Time, Dry, Wet;
    float Filter_F, Filter_Q, Filter_K, Filter_P, 
      Filter_XL, Filter_Y1L, Filter_Y2L, Filter_Y3L, Filter_Y4L, Filter_OldXL, Filter_OldY1L, Filter_OldY2L, Filter_OldY3L,
      Filter_XR, Filter_Y1R, Filter_Y2R, Filter_Y3R, Filter_Y4R, Filter_OldXR, Filter_OldY1R, Filter_OldY2R, Filter_OldY3R,
      Filter_LowL, Filter_LowR, Filter_BandL, Filter_BandR, Filter_HighL, Filter_HighR;
    unsigned int I;

    switch ( Type )
      {
        case SYNTH_FILTER_STATE_LOW12:
          Filter_LowL = Filter_LowR = Filter_BandL = Filter_BandR = Filter_HighL = Filter_HighR = 0.0f;
          for ( I=0; I<Buf->Length; I++ )
            {
              Time = (float) I / (float) SAMPLERATE;
              
              Wet = Synth_ControlValue( DryWetCtrl, Time );
              Dry = 1.0f - Wet;

              Filter_F = 2.0f * Synth_ControlValue( Cutoff, Time ) / (float) SAMPLERATE;
              Filter_Q = Synth_ControlValue( Resonance, Time );

              Filter_LowL += Filter_F * Filter_BandL;
              Filter_HighL = Filter_Q * Buf->Ptr[ 2*I+0 ] - Filter_LowL - Filter_Q * Filter_BandL;
              Filter_BandL += Filter_F * Filter_HighL; 
              Buf->Ptr[ 2*I+0 ] = Dry * Buf->Ptr[ 2*I+0 ] + Wet * Filter_LowL;

              Filter_LowR += Filter_F * Filter_BandR;
              Filter_HighR = Filter_Q * Buf->Ptr[ 2*I+1 ] - Filter_LowR - Filter_Q * Filter_BandR;
              Filter_BandR += Filter_F * Filter_HighR; 
              Buf->Ptr[ 2*I+1 ] = Dry * Buf->Ptr[ 2*I+1 ] + Wet * Filter_LowR;
             }
          break;

        case SYNTH_FILTER_STATE_BAND12:
          Filter_LowL = Filter_LowR = Filter_BandL = Filter_BandR = Filter_HighL = Filter_HighR = 0.0f;
          for ( I=0; I<Buf->Length; I++ )
            {
              Time = (float) I / (float) SAMPLERATE;

              Wet = Synth_ControlValue( DryWetCtrl, Time );
              Dry = 1.0f - Wet;

              Filter_F = 2.0f * Synth_ControlValue( Cutoff, Time ) / (float) SAMPLERATE;
              Filter_Q = Synth_ControlValue( Resonance, Time );

              Filter_LowL += Filter_F * Filter_BandL;
              Filter_HighL = Filter_Q * Buf->Ptr[ 2*I+0 ] - Filter_LowL - Filter_Q * Filter_BandL;
              Filter_BandL += Filter_F * Filter_HighL; 
              Buf->Ptr[ 2*I+0 ] = Dry * Buf->Ptr[ 2*I+0 ] + Wet * Filter_BandL;

              Filter_LowR += Filter_F * Filter_BandR;
              Filter_HighR = Filter_Q * Buf->Ptr[ 2*I+1 ] - Filter_LowR - Filter_Q * Filter_BandR;
              Filter_BandR += Filter_F * Filter_HighR; 
              Buf->Ptr[ 2*I+1 ] = Dry * Buf->Ptr[ 2*I+1 ] + Wet * Filter_BandR;
             }
          break;

        case SYNTH_FILTER_STATE_HIGH12:
          Filter_LowL = Filter_LowR = Filter_BandL = Filter_BandR = Filter_HighL = Filter_HighR = 0.0f;
          for ( I=0; I<Buf->Length; I++ )
            {
              Time = (float) I / (float) SAMPLERATE;

              Wet = Synth_ControlValue( DryWetCtrl, Time );
              Dry = 1.0f - Wet;

              Filter_F = 2.0f * Synth_ControlValue( Cutoff, Time ) / (float) SAMPLERATE;
              Filter_Q = Synth_ControlValue( Resonance, Time );

              Filter_LowL += Filter_F * Filter_BandL;
              Filter_HighL = Filter_Q * Buf->Ptr[ 2*I+0 ] - Filter_LowL - Filter_Q * Filter_BandL;
              Filter_BandL += Filter_F * Filter_HighL; 
              Buf->Ptr[ 2*I+0 ] = Dry * Buf->Ptr[ 2*I+0 ] + Wet * Filter_HighL;

              Filter_LowR += Filter_F * Filter_BandR;
              Filter_HighR = Filter_Q * Buf->Ptr[ 2*I+1 ] - Filter_LowR - Filter_Q * Filter_BandR;
              Filter_BandR += Filter_F * Filter_HighR; 
              Buf->Ptr[ 2*I+1 ] = Dry * Buf->Ptr[ 2*I+1 ] + Wet * Filter_HighR;
             }
          break;

        case SYNTH_FILTER_STATE_NOTCH12:
          Filter_LowL = Filter_LowR = Filter_BandL = Filter_BandR = Filter_HighL = Filter_HighR = 0.0f;
          for ( I=0; I<Buf->Length; I++ )
            {
              Time = (float) I / (float) SAMPLERATE;

              Wet = Synth_ControlValue( DryWetCtrl, Time );
              Dry = 1.0f - Wet;

              Filter_F = 2.0f * Synth_ControlValue( Cutoff, Time ) / (float) SAMPLERATE;
              Filter_Q = Synth_ControlValue( Resonance, Time );

              Filter_LowL += Filter_F * Filter_BandL;
              Filter_HighL = Filter_Q * Buf->Ptr[ 2*I+0 ] - Filter_LowL - Filter_Q * Filter_BandL;
              Filter_BandL += Filter_F * Filter_HighL; 
              Buf->Ptr[ 2*I+0 ] = Dry * Buf->Ptr[ 2*I+0 ] + Wet * ( Filter_HighL + Filter_LowL );

              Filter_LowR += Filter_F * Filter_BandR;
              Filter_HighR = Filter_Q * Buf->Ptr[ 2*I+1 ] - Filter_LowR - Filter_Q * Filter_BandR;
              Filter_BandR += Filter_F * Filter_HighR; 
              Buf->Ptr[ 2*I+1 ] = Dry * Buf->Ptr[ 2*I+1 ] + Wet * ( Filter_HighR + Filter_LowR );
             }
          break;

        case SYNTH_FILTER_MOOG1_LOW24:

          Filter_Y1L = Filter_Y2L = Filter_Y3L = Filter_Y4L = Filter_OldXL = Filter_OldY1L = Filter_OldY1L = Filter_OldY2L = Filter_OldY3L = 0.0f;
          Filter_Y1R = Filter_Y2R = Filter_Y3R = Filter_Y4R = Filter_OldXR = Filter_OldY1R = Filter_OldY1R = Filter_OldY2R = Filter_OldY3R = 0.0f;
          for ( I=0; I<Buf->Length; I++ )
            {
              Time = (float) I / (float) SAMPLERATE;

              Wet = Synth_ControlValue( DryWetCtrl, Time );
              Dry = 1.0f - Wet;

              Filter_F = 2.0f * Synth_ControlValue( Cutoff, Time ) / (float) SAMPLERATE;
              // -- Empirical tuning:
              Filter_K =  - 1.6f * Fsqr( Filter_F ) + 3.6f * Filter_F - 1.0f; 
              Filter_P = ( Filter_K+1.0f ) * 0.5f;
              Filter_Q = Synth_ControlValue( Resonance, Time ) * Fexp(( 1.0f - Filter_P ) * 1.386249f );

              Filter_XL = Buf->Ptr[ 2*I+0 ] - Filter_Q * Filter_Y4L;
              Filter_Y1L = ( Filter_XL + Filter_OldXL ) * Filter_P - Filter_K * Filter_Y1L;
              Filter_Y2L = ( Filter_Y1L + Filter_OldY1L ) * Filter_P - Filter_K * Filter_Y2L;
              Filter_Y3L = ( Filter_Y2L + Filter_OldY2L ) * Filter_P - Filter_K * Filter_Y3L;
              Filter_Y4L = ( Filter_Y3L + Filter_OldY3L ) * Filter_P - Filter_K * Filter_Y4L;
              // -- Clipper band limited sigmoid.
              Filter_Y4L = Filter_Y4L * ( 1.0f - Fsqr( Filter_Y4L ) / 6.0f );
              Filter_OldXL = Filter_XL;
              Filter_OldY1L = Filter_Y1L;
              Filter_OldY2L = Filter_Y2L;
              Filter_OldY3L = Filter_Y3L;
              Buf->Ptr[ 2*I+0 ] = Dry * Buf->Ptr[ 2*I+0 ] + Wet * Filter_Y4L;

              Filter_XR = Buf->Ptr[ 2*I+1 ] - Filter_Q * Filter_Y4R;
              Filter_Y1R = ( Filter_XR + Filter_OldXR ) * Filter_P - Filter_K * Filter_Y1R;
              Filter_Y2R = ( Filter_Y1R + Filter_OldY1R ) * Filter_P - Filter_K * Filter_Y2R;
              Filter_Y3R = ( Filter_Y2R + Filter_OldY2R ) * Filter_P - Filter_K * Filter_Y3R;
              Filter_Y4R = ( Filter_Y3R + Filter_OldY3R ) * Filter_P - Filter_K * Filter_Y4R;
              // -- Clipper band limited sigmoid.
              Filter_Y4R = Filter_Y4R * ( 1.0f - Fsqr( Filter_Y4R ) / 6.0f );
              Filter_OldXR = Filter_XR;
              Filter_OldY1R = Filter_Y1R;
              Filter_OldY2R = Filter_Y2R;
              Filter_OldY3R = Filter_Y3R;
              Buf->Ptr[ 2*I+1 ] = Dry * Buf->Ptr[ 2*I+1 ] + Wet * Filter_Y4R;
             }          
          break;
       }
   }

// ---------------------------------------------
//    R E V E R B
// ---------------------------------------------

#define UNDENORMALIZE(Sample) if(((*(unsigned int*)&Sample)&0x7f800000)==0) Sample=0.0f

float CombFilterStore[16], *BufComb[16], *BufAllPass[8];

int BufCombIdx[16], BufCombSize[16] = 
  { 1116, 1116+23, 1188, 1188+23, 1277, 1277+23, 1356, 1356+23, 1422, 1422+23, 1491, 1491+23, 1557, 1557+23, 1617, 1617+23 };
int BufAllPassIdx[8], BufAllPassSize[8] =
  { 556, 556+23, 441, 441+23, 341, 341+23, 225, 225+23 };

// ---------------------------------------------

float Synth_Reverb_Comb( int Num, float Input, float Damp, float FeedBack )
  {
    float Output;

    Output = BufComb[ Num ][ BufCombIdx[ Num ]];
    UNDENORMALIZE( Output );

    CombFilterStore[ Num ] = CombFilterStore[ Num ] * Damp + Output * ( 1.0f - Damp );
    UNDENORMALIZE( CombFilterStore[ Num ] );

    BufComb[ Num ][ BufCombIdx[ Num ]] = Input + CombFilterStore[ Num ] * FeedBack;

    if (( ++( BufCombIdx[ Num ] )) >= ( BufCombSize[ Num ] )) BufCombIdx[ Num ] = 0;

    return Output;
   }

// ---------------------------------------------

float Synth_Reverb_AllPass( int Num, float Input, float FeedBack )
  {
    float BufOut, Output;

    BufOut = BufAllPass[ Num ][ BufAllPassIdx[ Num ]];
    UNDENORMALIZE( BufOut );

    Output = BufOut - Input;

    BufAllPass[ Num ][ BufAllPassIdx[ Num ]] = Input + BufOut * FeedBack;

    if ( ++BufAllPassIdx[ Num ] >= BufAllPassSize[ Num ] ) BufAllPassIdx[ Num ] = 0;

    return Output;
   }

// ---------------------------------------------

void Synth_Reverb( SYNTH_BUFFER *Buf, SYNTH_CONTROL *RoomSizeCtrl, SYNTH_CONTROL *WidthCtrl, SYNTH_CONTROL *DampCtrl, SYNTH_CONTROL *FeedBackCtrl, SYNTH_CONTROL *DryWetCtrl )
  {
    unsigned int I, J;
    float OutL, OutR, Input, Time, RoomSize, Width, Damp, FeedBack, DryWet, Dry, Wet1, Wet2;

    for ( I=0; I<16; I++ )
      { 
        BufCombIdx[ I ] = 0;
        BufComb[ I ] = (float*) HeapAlloc( GetProcessHeap(), HEAP_ZERO_MEMORY, sizeof(float) * BufCombSize[ I ] );
        CombFilterStore[ I ] = 0.0f;
       }
    for ( I=0; I<8; I++ )
      {
        BufAllPassIdx[ I ] = 0;
        BufAllPass[ I ] = (float*) HeapAlloc( GetProcessHeap(), HEAP_ZERO_MEMORY, sizeof(float) * BufAllPassSize[ I ] );
       }

    // --- Process.

    for ( I=0; I<Buf->Length; I++ )
      {
        Time = (float) I / (float) Buf->Length;

        DryWet = Synth_ControlValue( DryWetCtrl, Time );
        Width = Synth_ControlValue( WidthCtrl, Time );

        Dry = 1.0f - DryWet;
        Wet1 = DryWet * ( 1.0f + Width ) * 0.5f;
        Wet2 = DryWet * ( 1.0f - Width ) * 0.5f;
        
        OutL = 0.0f;
        OutR = 0.0f;

        Input = ( Buf->Ptr[ 2*I+0 ] + Buf->Ptr[ 2*I+1 ] ) * 0.015f;

        RoomSize = Synth_ControlValue( RoomSizeCtrl, Time );
        Damp = Synth_ControlValue( DampCtrl, Time );

        for ( J=0; J<8; J++ )
          {
            OutL += Synth_Reverb_Comb( 2*J+0, Input, 0.4f * Damp, 0.7f + 0.28f * RoomSize );
            OutR += Synth_Reverb_Comb( 2*J+1, Input, 0.4f * Damp, 0.7f + 0.28f * RoomSize );
           }

        FeedBack = Synth_ControlValue( FeedBackCtrl, Time );

        for ( J=0; J<4; J++ )
          {
            OutL = Synth_Reverb_AllPass( 2*J+0, OutL, 0.5f * FeedBack );
            OutR = Synth_Reverb_AllPass( 2*J+1, OutR, 0.5f * FeedBack );
           }
  
        Buf->Ptr[ 2*I+0 ] = Buf->Ptr[ 2*I+0 ] * Dry + OutL * Wet1 + OutR * Wet2;
        Buf->Ptr[ 2*I+1 ] = Buf->Ptr[ 2*I+1 ] * Dry + OutR * Wet1 + OutL * Wet2;
       }

    // --- Cleanup.

    for ( I=0; I<16; I++ )
      HeapFree( GetProcessHeap(), 0, (void*) BufComb[ I ] );
    for ( I=0; I<8; I++ )
      HeapFree( GetProcessHeap(), 0, (void*) BufAllPass[ I ] );
   }

// ---------------------------------------------
//    C H O R U S / F L A N G E
// ---------------------------------------------

void Synth_ChorusFlange( SYNTH_BUFFER *Buf, SYNTH_CONTROL *DelayCtrl, SYNTH_CONTROL *FeedBackCtrl, SYNTH_CONTROL *DryWetCtrl )
  {
    unsigned int I;
    float Time, Pos, Delay, FeedBack, Dry, Wet;
    SYNTH_BUFFER *Buf2;

    Buf2 = Synth_BufferCopy( Buf );

    for ( I=0; I<Buf->Length; I++ )
      {
        Time = (float) I / (float) Buf->Length;
        Pos = (float) I;

        Delay = SAMPLERATE * Synth_ControlValue( DelayCtrl, Time );
        FeedBack = Synth_ControlValue( FeedBackCtrl, Time );
        Wet = 0.5f * Synth_ControlValue( DryWetCtrl, Time );
        Dry = 1.0f - Wet;

        Buf->Ptr[ I*2+0 ] = Dry * Buf->Ptr[ I*2+0 ] + Wet * ( Synth_BufferValue( Buf2, Pos - Delay, 0 ) + FeedBack * Synth_BufferValue( Buf, Pos - Delay, 0 ));
        Buf->Ptr[ I*2+1 ] = Dry * Buf->Ptr[ I*2+1 ] + Wet * ( Synth_BufferValue( Buf2, Pos - Delay, 1 ) + FeedBack * Synth_BufferValue( Buf, Pos - Delay, 1 ));
       }

    Synth_BufferCleanup( Buf2 );
   }

